
import React, { useState } from 'react';
import { IntegratedInput } from './CommonUI.tsx';
import { router } from '@inertiajs/react';
import { 
    Building2, MapPin, Smartphone, Save, History, BadgeCheck, Lock, Unlock, 
    ShieldAlert, AlertTriangle, Eye, EyeOff, LayoutGrid, Info
} from 'lucide-react';

export function SettingsPage({ showroomSettings, notify, allModules }: any) {
    const [isSaving, setIsSaving] = useState(false);
    const [localState, setLocalState] = useState(showroomSettings);

    const handleSaveSettings = () => {
        setIsSaving(true);
        router.post('/settings', localState, {
            onFinish: () => {
                setIsSaving(false);
                if (notify) notify('System Configuration Updated Successfully');
            }
        });
    };

    const toggleModuleVisibility = (moduleId: string) => {
        if (moduleId === 'settings' || moduleId === 'dashboard') return;

        const hidden = localState.hiddenModules || [];
        const newHidden = hidden.includes(moduleId) 
            ? hidden.filter((id: string) => id !== moduleId) 
            : [...hidden, moduleId];

        setLocalState({ ...localState, hiddenModules: newHidden });
    };

    const toggleGuardrail = (field: 'disableEditing' | 'disableDeleting') => {
        setLocalState({ ...localState, [field]: !localState[field] });
    };

    return (
        <div className="animate-fade-in space-y-10 pb-20">
            <div className="px-4 lg:px-0 flex justify-between items-end">
                <div>
                    <h3 className="text-lg font-bold font-heading text-slate-900 uppercase italic">Control Panel</h3>
                    <p className="text-[10px] text-slate-400 font-bold uppercase tracking-widest mt-1.5">Master System Configuration</p>
                </div>
                <div className="hidden lg:block">
                     <p className="text-[8px] font-black text-emerald-500 uppercase tracking-[0.3em] bg-emerald-50 px-3 py-1 rounded-full border border-emerald-100 flex items-center gap-2">
                        <BadgeCheck size={12} /> Full Access Mode
                     </p>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 px-4 lg:px-0">
                <div className="lg:col-span-2 space-y-8">
                    <div className="bg-white rounded-[2.5rem] border border-slate-200 shadow-sm overflow-hidden">
                        <div className="bg-slate-900 p-8 flex justify-between items-center text-white">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-white/10 rounded-2xl flex items-center justify-center text-cyan-400 shadow-inner">
                                    <Building2 size={24} />
                                </div>
                                <div className="leading-none">
                                    <h4 className="font-black uppercase text-xs tracking-widest italic leading-none">Store Identity</h4>
                                    <p className="text-[9px] text-slate-400 font-bold uppercase mt-1.5 leading-none">Primary Branding Specs</p>
                                </div>
                            </div>
                            {isSaving ? (
                                <div className="flex items-center gap-3 px-6 py-2.5 bg-slate-800 rounded-xl text-[10px] font-black uppercase text-slate-400 tracking-widest">
                                    <History size={14} className="animate-spin text-cyan-400"/> Processing...
                                </div>
                            ) : (
                                <button onClick={handleSaveSettings} className="px-6 py-2.5 bg-cyan-600 hover:bg-cyan-500 rounded-xl flex items-center gap-2 text-[10px] font-black uppercase tracking-widest transition-all shadow-lg shadow-cyan-500/20 active:scale-95">
                                    <Save size={14}/> Save Changes
                                </button>
                            )}
                        </div>

                        <div className="p-10 space-y-8">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div className="md:col-span-2">
                                    <IntegratedInput 
                                        label="Registered Showroom Name" 
                                        value={localState.showroomName} 
                                        onChange={(e: any) => setLocalState({...localState, showroomName: e.target.value})} 
                                        icon={Building2} 
                                        placeholder="Business Name"
                                    />
                                </div>
                                <div className="md:col-span-2">
                                    <IntegratedInput 
                                        label="Official Physical Address" 
                                        value={localState.address} 
                                        onChange={(e: any) => setLocalState({...localState, address: e.target.value})} 
                                        icon={MapPin} 
                                        placeholder="Full address"
                                    />
                                </div>
                                <IntegratedInput 
                                    label="Primary Contact Number" 
                                    value={localState.phone} 
                                    onChange={(e: any) => setLocalState({...localState, phone: e.target.value})} 
                                    icon={Smartphone} 
                                    placeholder="03xx-xxxxxxx"
                                />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white rounded-[2.5rem] border border-slate-200 shadow-sm overflow-hidden">
                        <div className="p-8 border-b border-slate-100 flex items-center justify-between bg-slate-50/50">
                            <div className="flex items-center gap-3">
                                <LayoutGrid size={20} className="text-cyan-500" />
                                <h4 className="text-xs font-black uppercase tracking-widest text-slate-900 italic">Interface Management</h4>
                            </div>
                        </div>
                        <div className="p-8">
                            <p className="text-[10px] font-bold text-slate-400 uppercase mb-8 flex items-center gap-2">
                                <Info size={14} className="text-indigo-400" />
                                Changes will reflect after saving.
                            </p>
                            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                                {allModules.map((module: any) => {
                                    const isHidden = (localState.hiddenModules || []).includes(module.id);
                                    const isCritical = module.id === 'settings' || module.id === 'dashboard';
                                    
                                    return (
                                        <div 
                                            key={module.id} 
                                            onClick={() => !isCritical && toggleModuleVisibility(module.id)}
                                            className={`p-4 rounded-2xl border-2 transition-all duration-300 flex flex-col gap-3 group relative cursor-pointer ${
                                                isHidden ? 'bg-slate-50 border-slate-100 opacity-60' : 'bg-white border-slate-100 hover:border-cyan-200 shadow-sm'
                                            }`}
                                        >
                                            <div className="flex justify-between items-start">
                                                <div className={`p-2 rounded-lg ${isHidden ? 'bg-slate-200 text-slate-400' : 'bg-slate-900 text-white shadow-md'}`}>
                                                    <module.icon size={16} />
                                                </div>
                                                <div className={`p-1.5 rounded-full ${isHidden ? 'text-rose-400 bg-rose-50' : 'text-emerald-500 bg-emerald-50'}`}>
                                                    {isHidden ? <EyeOff size={12} /> : <Eye size={12} />}
                                                </div>
                                            </div>
                                            <div className="leading-none">
                                                <p className={`text-[10px] font-black uppercase tracking-tight ${isHidden ? 'text-slate-400' : 'text-slate-900'}`}>{module.label}</p>
                                                {isCritical && <p className="text-[7px] font-black text-rose-500 uppercase mt-1">Core System</p>}
                                            </div>
                                        </div>
                                    );
                                })}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
