
import React, { useState, useMemo } from 'react';
import {
    Printer, Search, X, Pencil, Trash2, Factory, Tag, Calendar, Palette,
    StickyNote, DollarSign, Hash, LayoutGrid, Settings2, Plus,
    Layers, Package, CheckCircle2, Bike
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export function ProductsPage({
                                 products, makes, years, colors,
                                 disableEditing, disableDeleting, notify
                             }: any) {
    const [activeTab, setActiveTab] = useState<'catalog' | 'attributes'>('catalog');
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingId, setEditingId] = useState(null);
    const [searchTerm, setSearchTerm] = useState('');

    const [newMake, setNewMake] = useState('');
    const [newYear, setNewYear] = useState('');
    const [newColor, setNewColor] = useState('');

    const [formData, setFormData] = useState({
        make: makes[0] || '',
        model: '',
        year: years[0] || '',
        color: colors[0] || '',
        price: '',
        cost: '',
        notes: '',
        date: getCurrentDate(),
        refNumber: ''
    });

    const filteredProducts = useMemo(() => {
        if (!searchTerm) return products;
        const lower = searchTerm.toLowerCase();
        return products.filter((p: any) =>
            p.make.toLowerCase().includes(lower) ||
            p.model.toLowerCase().includes(lower) ||
            p.year.toLowerCase().includes(lower) ||
            p.color.toLowerCase().includes(lower) ||
            (p.refNumber && p.refNumber.toLowerCase().includes(lower))
        );
    }, [products, searchTerm]);

    const handleSave = () => {
        if(!formData.model || !formData.price) return;

        const url = editingId ? `/products/${editingId}` : '/products';
        // Using standard Inertia visit via router
        // In production, we'd use PUT/PATCH for updates, but using POST for simplicity in this monolithic refactor
        const data = editingId ? { ...formData, id: editingId } : formData;

        router.post(url, data, {
            onSuccess: () => {
                notify(editingId ? 'Product Updated' : 'New Model Added to Catalog');
                closeModal();
            }
        });
    };

    const handleDelete = (id: string, name: string) => {
        if(confirm(`Delete ${name} from catalog?`)) {
            router.delete(`/products/${id}`, {
                onSuccess: () => notify('Product Removed')
            });
        }
    };

    const addAttribute = (type: string, value: string) => {
        if (!value) return;
        router.post('/attributes', { type, value }, {
            onSuccess: () => {
                notify(`${type.charAt(0).toUpperCase() + type.slice(1)} Added Successfully`);
                if (type === 'make') setNewMake('');
                if (type === 'year') setNewYear('');
                if (type === 'color') setNewColor('');
            }
        });
    };

    const openEditModal = (product: any) => {
        setEditingId(product.id);
        setFormData({ ...product, price: String(product.price), cost: String(product.cost), refNumber: product.refNumber || '' });
        setIsModalOpen(true);
    };

    const closeModal = () => {
        setIsModalOpen(false);
        setEditingId(null);
        setFormData({ make: makes[0] || '', model: '', year: years[0] || '', color: colors[0] || '', price: '', cost: '', notes: '', date: getCurrentDate(), refNumber: '' });
    };

    return (
        <div className="animate-fade-in space-y-6">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 px-4 lg:px-0">
                <div className="flex p-1 bg-white border border-slate-200 rounded-2xl shadow-sm">
                    <button onClick={() => setActiveTab('catalog')} className={`flex items-center gap-2 px-6 py-2.5 rounded-xl text-xs font-black uppercase tracking-widest transition-all ${activeTab === 'catalog' ? 'bg-slate-900 text-white' : 'text-slate-400'}`}><LayoutGrid size={14}/> Unit Catalog</button>
                    <button onClick={() => setActiveTab('attributes')} className={`flex items-center gap-2 px-6 py-2.5 rounded-xl text-xs font-black uppercase tracking-widest transition-all ${activeTab === 'attributes' ? 'bg-slate-900 text-white' : 'text-slate-400'}`}><Settings2 size={14}/> Attribute Setup</button>
                </div>
                {activeTab === 'catalog' && (
                    <div className="flex gap-3 w-full sm:w-auto">
                        <button onClick={() => window.print()} className="flex-1 sm:flex-none flex items-center justify-center gap-2 px-6 py-2.5 bg-white border border-slate-200 text-slate-600 rounded-xl text-xs font-black uppercase tracking-widest hover:bg-slate-50"><Printer size={14} /> Print</button>
                        <button onClick={() => setIsModalOpen(true)} className="flex-1 sm:flex-none flex items-center justify-center gap-2 px-8 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-black uppercase tracking-widest shadow-xl hover:bg-black"><Plus size={14} /> New Model</button>
                    </div>
                )}
            </div>

            {activeTab === 'catalog' ? (
                <>
                    <div className="px-4 lg:px-0 no-print">
                        <div className="bg-white rounded-2xl border border-slate-200 flex items-center px-6 py-3 shadow-sm">
                            <Search size={18} className="text-slate-400 mr-4" />
                            <input type="text" placeholder="Search catalog..." className="bg-transparent border-none outline-none text-[11px] font-black w-full text-slate-900 uppercase tracking-widest" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                        </div>
                    </div>
                    <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0 print-table">
                        <table className="w-full text-left border-collapse">
                            <thead className="bg-slate-50 border-b border-slate-100 uppercase italic">
                            <tr className="text-[10px] font-bold text-slate-500"><th className="px-8 py-5">Product Master Info</th><th className="px-8 py-5">System Metadata</th><th className="px-8 py-5 text-right">Valuation</th><th className="px-8 py-5 text-right no-print">Actions</th></tr>
                            </thead>
                            <tbody className="divide-y divide-slate-50">
                            {filteredProducts.map((p: any) => (
                                <tr key={p.id} className="hover:bg-slate-50/50 transition-colors group">
                                    <td className="px-8 py-5">
                                        <div className="flex items-center gap-4">
                                            <div className="w-10 h-10 rounded-xl bg-slate-100 flex items-center justify-center text-slate-400"><Bike size={20}/></div>
                                            <div><p className="font-black text-slate-900 text-sm uppercase">{p.make} {p.model}</p><p className="text-[10px] font-bold text-slate-400 uppercase mt-1">{p.year} • {p.color}</p></div>
                                        </div>
                                    </td>
                                    <td className="px-8 py-5"><p className="text-[10px] font-bold text-slate-400 uppercase">Added: {p.date}</p></td>
                                    <td className="px-8 py-5 text-right"><p className="font-mono text-sm font-black text-emerald-600">{formatCurrency(p.price)}</p></td>
                                    <td className="px-8 py-5 text-right no-print">
                                        <div className="flex justify-end gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
                                            {!disableEditing && <button onClick={() => openEditModal(p)} className="p-2.5 text-slate-400 hover:text-cyan-600"><Pencil size={16} /></button>}
                                            {!disableDeleting && <button onClick={() => handleDelete(p.id, p.model)} className="p-2.5 text-slate-400 hover:text-rose-600"><Trash2 size={16} /></button>}
                                        </div>
                                    </td>
                                </tr>
                            ))}
                            </tbody>
                        </table>
                    </div>
                </>
            ) : (
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 px-4 lg:px-0">
                    {/* MAKES / BRANDS */}
                    <div className="bg-white rounded-[2.5rem] border border-slate-200 p-8 shadow-sm space-y-6">
                        <div className="flex items-center gap-3 border-b border-slate-50 pb-4"><Factory size={20} className="text-cyan-500" /><h4 className="text-xs font-black uppercase text-slate-900">Brand Registry</h4></div>
                        <div className="flex gap-2"><input type="text" placeholder="New Brand..." className="flex-1 bg-slate-50 border border-slate-200 rounded-xl px-4 py-2 text-xs font-bold uppercase outline-none focus:border-cyan-500 transition-all" value={newMake} onChange={(e) => setNewMake(e.target.value)} /><button onClick={() => addAttribute('make', newMake)} className="p-2.5 bg-slate-900 text-white rounded-xl hover:bg-black transition-all"><Plus size={18}/></button></div>
                        <div className="space-y-2 max-h-96 overflow-y-auto custom-scrollbar pr-2">{makes.map((m: string) => <div key={m} className="p-3 bg-slate-50 rounded-xl border border-slate-100 text-[10px] font-black uppercase text-slate-600">{m}</div>)}</div>
                    </div>

                    {/* YEARS */}
                    <div className="bg-white rounded-[2.5rem] border border-slate-200 p-8 shadow-sm space-y-6">
                        <div className="flex items-center gap-3 border-b border-slate-50 pb-4"><Calendar size={20} className="text-indigo-500" /><h4 className="text-xs font-black uppercase text-slate-900">Model Years</h4></div>
                        <div className="flex gap-2"><input type="text" placeholder="New Year..." className="flex-1 bg-slate-50 border border-slate-200 rounded-xl px-4 py-2 text-xs font-bold uppercase outline-none focus:border-indigo-500 transition-all" value={newYear} onChange={(e) => setNewYear(e.target.value)} /><button onClick={() => addAttribute('year', newYear)} className="p-2.5 bg-slate-900 text-white rounded-xl hover:bg-black transition-all"><Plus size={18}/></button></div>
                        <div className="space-y-2 max-h-96 overflow-y-auto custom-scrollbar pr-2">{years.map((y: string) => <div key={y} className="p-3 bg-slate-50 rounded-xl border border-slate-100 text-[10px] font-black uppercase text-slate-600">{y}</div>)}</div>
                    </div>

                    {/* COLORS */}
                    <div className="bg-white rounded-[2.5rem] border border-slate-200 p-8 shadow-sm space-y-6">
                        <div className="flex items-center gap-3 border-b border-slate-50 pb-4"><Palette size={20} className="text-rose-500" /><h4 className="text-xs font-black uppercase text-slate-900">Color Variants</h4></div>
                        <div className="flex gap-2"><input type="text" placeholder="New Color..." className="flex-1 bg-slate-50 border border-slate-200 rounded-xl px-4 py-2 text-xs font-bold uppercase outline-none focus:border-rose-500 transition-all" value={newColor} onChange={(e) => setNewColor(e.target.value)} /><button onClick={() => addAttribute('color', newColor)} className="p-2.5 bg-slate-900 text-white rounded-xl hover:bg-black transition-all"><Plus size={18}/></button></div>
                        <div className="space-y-2 max-h-96 overflow-y-auto custom-scrollbar pr-2">{colors.map((c: string) => <div key={c} className="p-3 bg-slate-50 rounded-xl border border-slate-100 text-[10px] font-black uppercase text-slate-600">{c}</div>)}</div>
                    </div>
                </div>
            )}

            {isModalOpen && (
                <div className="fixed inset-0 bg-slate-900/60 z-[100] flex items-center justify-center p-4 backdrop-blur-sm">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg p-10 space-y-8 shadow-2xl my-auto animate-fade-in">
                        <div className="flex items-center justify-between border-b border-slate-100 pb-6"><h3 className="font-black uppercase italic text-sm tracking-widest">{editingId ? 'Modify Unit' : 'New Model Registration'}</h3><button onClick={closeModal}><X size={20} /></button></div>
                        <div className="grid grid-cols-2 gap-5">
                            <IntegratedSelect label="Select Make" value={formData.make} onChange={(e: any) => setFormData({...formData, make: e.target.value})} options={makes} icon={Factory} />
                            <IntegratedInput label="Model Name" value={formData.model} onChange={(e: any) => setFormData({...formData, model: e.target.value})} placeholder="e.g. CD 70" icon={Tag} />
                            <IntegratedSelect label="Mfg Year" value={formData.year} onChange={(e: any) => setFormData({...formData, year: e.target.value})} options={years} icon={Calendar} />
                            <IntegratedSelect label="Color Variant" value={formData.color} onChange={(e: any) => setFormData({...formData, color: e.target.value})} options={colors} icon={Palette} />
                            <div className="col-span-2 grid grid-cols-2 gap-5 p-6 bg-slate-50 rounded-[2rem]">
                                <IntegratedInput label="Retail Price" value={formData.price} onChange={(e: any) => setFormData({...formData, price: e.target.value})} icon={DollarSign} />
                                <IntegratedInput label="Cost Price" value={formData.cost} onChange={(e: any) => setFormData({...formData, cost: e.target.value})} icon={Layers} />
                            </div>
                        </div>
                        <button onClick={handleSave} className="w-full bg-slate-900 text-white py-5 rounded-[2rem] font-black uppercase text-xs tracking-widest hover:bg-black transition-all"><CheckCircle2 size={20} className="inline mr-2" /> Commit to Catalog</button>
                    </div>
                </div>
            )}
        </div>
    );
}
