
import React, { useState, useMemo } from 'react';
import {
    CreditCard, HandCoins, Search, TrendingUp, User, BadgeCheck, DollarSign,
    Calendar, Hash, Landmark, FileText, CheckCircle2, AlertTriangle, Clock,
    ChevronRight, X, Bike, History, PieChart, ArrowUpRight
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export function InstallmentsPage({ salesHistory, accounts, notify }: any) {
    const [searchTerm, setSearchTerm] = useState('');
    const [activeTab, setActiveTab] = useState<'active' | 'settled'>('active');
    const [isPaymentModalOpen, setIsPaymentModalOpen] = useState(false);
    const [viewingLease, setViewingLease] = useState<any>(null); // For detailed modal

    // Payment Form State
    const [paymentForm, setPaymentForm] = useState({
        saleId: '',
        amount: '',
        date: getCurrentDate(),
        accountId: 'ACC-001',
        note: ''
    });

    // --- LOGIC: Process Sales Data for Lease Tracking ---
    const leasePortfolio = useMemo(() => {
        return (salesHistory || [])
            // ROBUST FILTER: Include anything that is NOT explicitly 'Full Payment'.
            .filter((s: any) => s && s.paymentType !== 'Full Payment')
            .map((s: any) => {
                const paid = (s.payments || []).reduce((a:number, p:any) => a + (Number(p.amount) || 0), 0);
                const total = Number(s.total) || 0;
                const balance = total - paid;
                const progress = total > 0 ? Math.min(100, (paid / total) * 100) : 0;

                // --- SAFE DATE PARSING ---
                let saleDate = new Date(s.date);
                if (isNaN(saleDate.getTime())) {
                    // Fallback to today if date is invalid to prevent crash
                    saleDate = new Date();
                }

                const today = new Date();
                const dayOfMonth = saleDate.getDate();

                // Calculate Next Due Date (Same day of current month)
                // JS Date handles overflow (e.g. Feb 30 -> Mar 2) automatically
                let nextDue = new Date(today.getFullYear(), today.getMonth(), dayOfMonth);

                // If calculated nextDue is invalid (rare), reset to today
                if (isNaN(nextDue.getTime())) nextDue = new Date();

                if (nextDue < today) {
                    // If day has passed, move to next month
                    nextDue = new Date(today.getFullYear(), today.getMonth() + 1, dayOfMonth);
                }

                // Safe ISO String conversion
                const nextDueDateStr = !isNaN(nextDue.getTime()) ? nextDue.toISOString().split('T')[0] : getCurrentDate();

                // Status Logic
                const expectedMonthly = Number(s.monthlyAmount) || 0;
                let monthsElapsed = 0;
                if (!isNaN(saleDate.getTime())) {
                    monthsElapsed = Math.max(1, (today.getFullYear() - saleDate.getFullYear()) * 12 + (today.getMonth() - saleDate.getMonth()));
                }

                const expectedPaid = (Number(s.downPayment) || 0) + (monthsElapsed * expectedMonthly);

                let status = 'Good';
                if (balance <= 0) status = 'Settled';
                else if (paid < expectedPaid - (expectedMonthly * 0.5)) status = 'Overdue'; // Tolerance of half a month
                else if (paid < expectedPaid) status = 'Due Soon';

                // Robust Customer Name Extraction
                const custName = s.customer ? (typeof s.customer === 'object' ? s.customer.name : s.customer) : 'Unknown Client';
                const custPhone = s.customer && typeof s.customer === 'object' ? s.customer.contact : '---';

                return {
                    ...s,
                    paid,
                    balance,
                    progress,
                    nextDueDate: nextDueDateStr,
                    status,
                    customerName: custName,
                    phone: custPhone
                };
            })
            // SORT: Newest First
            .sort((a: any, b: any) => new Date(b.date || 0).getTime() - new Date(a.date || 0).getTime());
    }, [salesHistory]);

    const filteredLeases = useMemo(() => {
        return leasePortfolio.filter((s: any) => {
            const matchesSearch = s.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                (s.bike || '').toLowerCase().includes(searchTerm.toLowerCase()) ||
                (s.refNumber && s.refNumber.toLowerCase().includes(searchTerm.toLowerCase()));

            const matchesTab = activeTab === 'active' ? s.balance > 0 : s.balance <= 0;
            return matchesSearch && matchesTab;
        });
    }, [leasePortfolio, searchTerm, activeTab]);

    const stats = useMemo(() => {
        const active = leasePortfolio.filter((l: any) => l.balance > 0);
        const totalOutstanding = active.reduce((a: number, l: any) => a + l.balance, 0);
        const overdueCount = active.filter((l: any) => l.status === 'Overdue').length;
        const collectionRate = active.length > 0 ? active.reduce((a: number, l: any) => a + l.progress, 0) / active.length : 0;

        return { totalOutstanding, activeCount: active.length, overdueCount, collectionRate };
    }, [leasePortfolio]);

    // --- HANDLERS ---
    const openPaymentModal = (lease: any) => {
        setViewingLease(lease); // Also set detailed view context
        setPaymentForm({
            saleId: lease.id,
            amount: lease.monthlyAmount ? lease.monthlyAmount.toString() : '0', // Auto-fill monthly installment
            date: getCurrentDate(),
            accountId: accounts?.[0]?.id || '',
            note: 'Monthly Installment'
        });
        setIsPaymentModalOpen(true);
    };

    const handleReceivePayment = () => {
        if (!paymentForm.saleId || !paymentForm.amount) return;

        router.post('/payments', paymentForm, {
            onSuccess: () => {
                setIsPaymentModalOpen(false);
                notify('Installment Received Successfully');
            }
        });
    };

    return (
        <div className="animate-fade-in space-y-6">
            {/* Header & Stats */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl flex flex-col justify-between relative overflow-hidden group">
                    <div className="absolute right-0 top-0 p-4 opacity-10 group-hover:scale-110 transition-transform"><TrendingUp size={80}/></div>
                    <p className="text-[10px] font-black text-emerald-400 uppercase tracking-widest mb-1">Portfolio Balance</p>
                    <h3 className="text-2xl font-black font-mono tracking-tight">{formatCurrency(stats.totalOutstanding)}</h3>
                    <div className="mt-2 text-[9px] font-bold text-slate-400 uppercase flex items-center gap-1">
                        <User size={12}/> {stats.activeCount} Active Contracts
                    </div>
                </div>

                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-rose-50 text-rose-600 rounded-xl flex items-center justify-center shadow-inner"><AlertTriangle size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Overdue Accounts</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{stats.overdueCount}</p>
                        <p className="text-[8px] font-bold text-rose-500 uppercase">Require Follow-up</p>
                    </div>
                </div>

                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-indigo-50 text-indigo-600 rounded-xl flex items-center justify-center shadow-inner"><PieChart size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Collection Health</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{stats.collectionRate.toFixed(1)}%</p>
                        <p className="text-[8px] font-bold text-indigo-500 uppercase">Avg. Equity Owned</p>
                    </div>
                </div>

                <div className="bg-gradient-to-br from-emerald-500 to-emerald-600 rounded-[2rem] p-6 text-white shadow-lg flex flex-col justify-center items-center text-center cursor-pointer hover:shadow-emerald-500/30 transition-all" onClick={() => router.visit('/reports')}>
                    <FileText size={24} className="mb-2 opacity-80"/>
                    <p className="text-[10px] font-black uppercase tracking-widest">View Recovery Report</p>
                </div>
            </div>

            {/* Controls & Tabs */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div className="flex p-1 bg-white border border-slate-200 rounded-xl shadow-sm">
                    <button onClick={() => setActiveTab('active')} className={`px-6 py-2.5 rounded-lg text-xs font-black uppercase tracking-widest transition-all ${activeTab === 'active' ? 'bg-slate-900 text-white shadow-md' : 'text-slate-400 hover:bg-slate-50'}`}>Active Leases</button>
                    <button onClick={() => setActiveTab('settled')} className={`px-6 py-2.5 rounded-lg text-xs font-black uppercase tracking-widest transition-all ${activeTab === 'settled' ? 'bg-slate-900 text-white shadow-md' : 'text-slate-400 hover:bg-slate-50'}`}>Settled History</button>
                </div>
                <div className="relative w-full sm:w-80">
                    <Search className="absolute left-4 top-1/2 -translate-y-1/2 text-slate-400" size={16}/>
                    <input
                        type="text"
                        placeholder="Search Customer, Bike or Ref..."
                        className="w-full pl-11 pr-4 py-3 bg-white border border-slate-200 rounded-2xl text-[11px] font-black uppercase tracking-[0.1em] outline-none focus:border-indigo-500 shadow-sm transition-all"
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                    />
                </div>
            </div>

            {/* Main List */}
            <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                <table className="w-full text-left">
                    <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                    <tr>
                        <th className="px-8 py-6">Customer Profile</th>
                        <th className="px-8 py-6">Asset Details</th>
                        <th className="px-8 py-6 text-center">Plan Progress</th>
                        <th className="px-8 py-6 text-right">Outstanding</th>
                        <th className="px-8 py-6 text-right">Actions</th>
                    </tr>
                    </thead>
                    <tbody className="divide-y divide-slate-100">
                    {filteredLeases.map((l: any) => (
                        <tr key={l.id} className="hover:bg-slate-50/50 transition-colors group">
                            <td className="px-8 py-6">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-slate-100 flex items-center justify-center text-slate-400 font-bold text-xs group-hover:bg-indigo-50 group-hover:text-indigo-600 transition-colors">
                                        {l.customerName.charAt(0)}
                                    </div>
                                    <div>
                                        <p className="font-black text-slate-900 text-sm uppercase tracking-tight">{l.customerName}</p>
                                        <div className="flex items-center gap-2 mt-1">
                                                <span className={`px-2 py-0.5 rounded text-[8px] font-black uppercase border ${
                                                    l.status === 'Overdue' ? 'bg-rose-50 text-rose-600 border-rose-100' :
                                                        l.status === 'Due Soon' ? 'bg-amber-50 text-amber-600 border-amber-100' :
                                                            l.status === 'Settled' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' :
                                                                'bg-slate-100 text-slate-500 border-slate-200'
                                                }`}>{l.status}</span>
                                            <span className="text-[9px] font-bold text-slate-400 font-mono">{l.phone}</span>
                                        </div>
                                    </div>
                                </div>
                            </td>
                            <td className="px-8 py-6">
                                <p className="text-xs font-bold text-slate-700 uppercase">{l.bike}</p>
                                <p className="text-[9px] font-mono text-slate-400 font-bold uppercase mt-0.5">ENG: {l.engine}</p>
                                <p className="text-[9px] font-mono text-slate-400 font-bold uppercase">REF: {l.refNumber || l.id}</p>
                            </td>
                            <td className="px-8 py-6 text-center">
                                <div className="flex flex-col items-center gap-1.5 w-32 mx-auto">
                                    <div className="w-full h-1.5 bg-slate-100 rounded-full overflow-hidden">
                                        <div className={`h-full rounded-full ${l.status === 'Settled' ? 'bg-emerald-500' : 'bg-indigo-500'}`} style={{width: `${l.progress}%`}}></div>
                                    </div>
                                    <p className="text-[9px] font-bold text-slate-400 uppercase tracking-wider">{l.progress.toFixed(0)}% Paid</p>
                                </div>
                            </td>
                            <td className="px-8 py-6 text-right">
                                <p className="font-black font-mono text-sm text-slate-900">{formatCurrency(l.balance)}</p>
                                {l.balance > 0 && (
                                    <p className="text-[9px] font-bold text-slate-400 uppercase mt-0.5">Next: {l.nextDueDate}</p>
                                )}
                            </td>
                            <td className="px-8 py-6 text-right">
                                <div className="flex justify-end gap-2">
                                    <button onClick={() => setViewingLease(l)} className="p-2.5 bg-white border border-slate-200 text-slate-400 hover:text-indigo-600 hover:border-indigo-200 rounded-xl transition-all shadow-sm">
                                        <FileText size={14}/>
                                    </button>
                                    {l.balance > 0 && (
                                        <button onClick={() => openPaymentModal(l)} className="flex items-center gap-2 px-4 py-2.5 bg-slate-900 text-white rounded-xl text-[10px] font-black uppercase tracking-widest shadow-lg hover:bg-emerald-600 transition-all">
                                            <HandCoins size={14}/> Collect
                                        </button>
                                    )}
                                </div>
                            </td>
                        </tr>
                    ))}
                    {filteredLeases.length === 0 && (
                        <tr><td colSpan={5} className="p-20 text-center text-slate-300 text-xs font-bold uppercase italic tracking-widest">No matching lease records found</td></tr>
                    )}
                    </tbody>
                </table>
            </div>

            {/* DETAILED LEDGER & PAYMENT MODAL */}
            {(isPaymentModalOpen || viewingLease) && (
                <div className="fixed inset-0 bg-slate-950/80 z-[120] flex items-center justify-center p-4 backdrop-blur-md overflow-y-auto leading-none">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-5xl shadow-2xl my-auto animate-fade-in relative flex flex-col md:flex-row overflow-hidden max-h-[90vh]">

                        {/* Left Panel: Ledger & Info */}
                        <div className="flex-1 p-8 bg-slate-50 overflow-y-auto custom-scrollbar">
                            <div className="flex items-center gap-4 mb-8">
                                <div className="w-12 h-12 bg-white rounded-2xl flex items-center justify-center shadow-sm text-indigo-600"><FileText size={24}/></div>
                                <div>
                                    <h3 className="font-black text-lg text-slate-900 uppercase italic">Lease Ledger</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest">Contract #{viewingLease?.id}</p>
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4 mb-6">
                                <div className="p-4 bg-white rounded-2xl border border-slate-100 shadow-sm">
                                    <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest mb-1">Total Contract</p>
                                    <p className="text-lg font-black text-slate-900 font-mono">{formatCurrency(viewingLease?.total)}</p>
                                </div>
                                <div className="p-4 bg-white rounded-2xl border border-slate-100 shadow-sm">
                                    <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest mb-1">Down Payment</p>
                                    <p className="text-lg font-black text-emerald-600 font-mono">{formatCurrency(viewingLease?.downPayment)}</p>
                                </div>
                                <div className="p-4 bg-white rounded-2xl border border-slate-100 shadow-sm">
                                    <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest mb-1">Monthly Due</p>
                                    <p className="text-lg font-black text-indigo-600 font-mono">{formatCurrency(viewingLease?.monthlyAmount)}</p>
                                </div>
                                <div className="p-4 bg-white rounded-2xl border border-slate-100 shadow-sm">
                                    <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest mb-1">Plan Duration</p>
                                    <p className="text-lg font-black text-slate-900 font-mono">{viewingLease?.installmentsCount} Months</p>
                                </div>
                            </div>

                            <h4 className="text-[10px] font-black text-slate-400 uppercase tracking-widest border-b border-slate-200 pb-2 mb-4">Transaction History</h4>
                            <div className="space-y-2">
                                {(viewingLease?.payments || []).map((p: any) => (
                                    <div key={p.id} className="flex justify-between items-center p-3 bg-white rounded-xl border border-slate-100">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-lg bg-emerald-50 text-emerald-600 flex items-center justify-center"><CheckCircle2 size={14}/></div>
                                            <div>
                                                <p className="text-[10px] font-bold text-slate-900 uppercase">{p.note || 'Installment'}</p>
                                                <p className="text-[9px] font-bold text-slate-400">{p.date}</p>
                                            </div>
                                        </div>
                                        <span className="text-xs font-black font-mono text-emerald-600">+{formatCurrency(p.amount)}</span>
                                    </div>
                                ))}
                                {(!viewingLease?.payments || viewingLease.payments.length === 0) && (
                                    <p className="text-center text-[10px] font-bold text-slate-400 italic py-4">No payments recorded yet</p>
                                )}
                            </div>
                        </div>

                        {/* Right Panel: Action & Summary */}
                        <div className="w-full md:w-96 bg-white p-8 border-l border-slate-100 flex flex-col relative">
                            <button onClick={() => { setIsPaymentModalOpen(false); setViewingLease(null); }} className="absolute top-6 right-6 p-2 bg-slate-50 hover:bg-rose-50 text-slate-400 hover:text-rose-500 rounded-full transition-colors"><X size={18}/></button>

                            <div className="mb-8">
                                <h4 className="font-black text-slate-900 uppercase text-lg">{viewingLease?.customerName}</h4>
                                <p className="text-xs font-bold text-slate-500 uppercase mt-1 flex items-center gap-2"><Bike size={12}/> {viewingLease?.bike}</p>
                            </div>

                            <div className="flex-1 flex flex-col justify-center space-y-6">
                                <div className="text-center space-y-2">
                                    <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest">Outstanding Balance</p>
                                    <p className="text-4xl font-black text-slate-900 font-mono tracking-tighter">{formatCurrency(viewingLease?.balance)}</p>
                                    <div className="inline-flex items-center gap-1 px-3 py-1 bg-slate-100 rounded-full text-[9px] font-bold text-slate-500 uppercase">
                                        <History size={10}/> Last Pay: {viewingLease?.payments?.[0]?.date || 'N/A'}
                                    </div>
                                </div>

                                {viewingLease?.balance > 0 ? (
                                    <div className="bg-indigo-50/50 p-6 rounded-[2rem] border border-indigo-100 space-y-4">
                                        <div className="flex items-center gap-2 mb-2">
                                            <CreditCard size={16} className="text-indigo-600"/>
                                            <h5 className="font-black text-indigo-900 uppercase text-xs tracking-widest">Receive Payment</h5>
                                        </div>

                                        <IntegratedInput
                                            label="Amount (PKR)"
                                            value={paymentForm.amount}
                                            onChange={(e: any) => setPaymentForm({...paymentForm, amount: e.target.value})}
                                            icon={DollarSign}
                                        />
                                        <IntegratedInput
                                            label="Date"
                                            type="date"
                                            value={paymentForm.date}
                                            onChange={(e: any) => setPaymentForm({...paymentForm, date: e.target.value})}
                                            icon={Calendar}
                                        />
                                        <IntegratedSelect
                                            label="Deposit To"
                                            value={paymentForm.accountId}
                                            onChange={(e: any) => setPaymentForm({...paymentForm, accountId: e.target.value})}
                                            options={accounts}
                                            displayKey="name"
                                            icon={Landmark}
                                        />
                                        <IntegratedInput
                                            label="Note / Month"
                                            value={paymentForm.note}
                                            onChange={(e: any) => setPaymentForm({...paymentForm, note: e.target.value})}
                                            icon={FileText}
                                            placeholder="e.g. September Installment"
                                        />

                                        <button onClick={handleReceivePayment} className="w-full bg-slate-900 text-white py-4 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-emerald-600 transition-all flex items-center justify-center gap-2 mt-4">
                                            <CheckCircle2 size={16}/> Confirm Receipt
                                        </button>
                                    </div>
                                ) : (
                                    <div className="p-8 bg-emerald-50 rounded-[2rem] border border-emerald-100 text-center space-y-2">
                                        <div className="w-16 h-16 bg-emerald-100 text-emerald-600 rounded-full flex items-center justify-center mx-auto mb-4">
                                            <BadgeCheck size={32}/>
                                        </div>
                                        <h5 className="font-black text-emerald-800 uppercase">Lease Settled</h5>
                                        <p className="text-[10px] font-bold text-emerald-600 uppercase leading-relaxed">This contract has been fully paid off. No further action required.</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
