
import React, { useState, useMemo } from 'react';
import { router } from '@inertiajs/react';
import {
    StickyNote, DollarSign, Tag, Calendar, Landmark,
    Search, Plus, X, BadgeCheck, Info, Wallet, PieChart,
    ArrowDownRight, Filter, Receipt
} from 'lucide-react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

const EXPENSE_CATEGORIES = [
    'General', 'Rent', 'Salaries', 'Utilities', 'Marketing',
    'Maintenance', 'Taxes', 'Stationery', 'Refreshments', 'Commission', 'Logistics'
];

export function ExpensesPage({ expenses, accounts, notify }: any) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');
    const [categoryFilter, setCategoryFilter] = useState('All');

    const [formData, setFormData] = useState({
        description: '', amount: '', category: 'General',
        date: getCurrentDate(), accountId: accounts?.[0]?.id || '', notes: '', refNumber: ''
    });

    const stats = useMemo(() => {
        const total = (expenses || []).reduce((acc: number, e: any) => acc + (Number(e.amount) || 0), 0);
        const count = (expenses || []).length;

        // Calculate breakdown
        const breakdown: Record<string, number> = {};
        (expenses || []).forEach((e: any) => {
            breakdown[e.category] = (breakdown[e.category] || 0) + Number(e.amount);
        });

        const topCat = Object.entries(breakdown).sort((a, b) => b[1] - a[1])[0];

        return { total, count, topCategory: topCat ? topCat[0] : 'None', topAmount: topCat ? topCat[1] : 0 };
    }, [expenses]);

    const filteredExpenses = useMemo(() => {
        return (expenses || []).filter((e: any) => {
            const term = searchTerm.toLowerCase();
            const matchesSearch = e.description.toLowerCase().includes(term) ||
                (e.refNumber || '').toLowerCase().includes(term) ||
                (e.notes || '').toLowerCase().includes(term);
            const matchesCat = categoryFilter === 'All' || e.category === categoryFilter;
            return matchesSearch && matchesCat;
        }).sort((a: any, b: any) => new Date(b.date).getTime() - new Date(a.date).getTime());
    }, [expenses, searchTerm, categoryFilter]);

    const handleSave = () => {
        if (!formData.description || !formData.amount || !formData.accountId) return;
        router.post('/expenses', formData, {
            onSuccess: () => {
                setIsModalOpen(false);
                setFormData({
                    description: '', amount: '', category: 'General',
                    date: getCurrentDate(), accountId: accounts?.[0]?.id || '', notes: '', refNumber: ''
                });
                if(notify) notify('Expense Voucher Recorded');
            }
        });
    };

    return (
        <div className="animate-fade-in space-y-6">
            {/* Stats Dashboard */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl relative overflow-hidden group">
                    <div className="absolute right-0 top-0 p-4 opacity-10 group-hover:scale-110 transition-transform"><Wallet size={80}/></div>
                    <p className="text-[10px] font-black text-rose-400 uppercase tracking-widest mb-1">Total Operational Outflow</p>
                    <h3 className="text-2xl font-black font-mono tracking-tight">{formatCurrency(stats.total)}</h3>
                    <div className="mt-4 flex items-center gap-2 text-[9px] font-bold text-slate-400 uppercase">
                        <Receipt size={12}/> {stats.count} Vouchers Processed
                    </div>
                </div>

                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4 group hover:shadow-md transition-all">
                    <div className="w-14 h-14 bg-indigo-50 text-indigo-600 rounded-2xl flex items-center justify-center shadow-inner"><PieChart size={28}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Major Cost Center</p>
                        <p className="text-xl font-black text-slate-900 uppercase truncate max-w-[150px]">{stats.topCategory}</p>
                        <p className="text-[9px] font-bold text-indigo-500 uppercase font-mono">{formatCurrency(stats.topAmount)}</p>
                    </div>
                </div>

                <div className="bg-gradient-to-br from-rose-500 to-rose-600 rounded-[2rem] p-6 text-white shadow-lg flex flex-col justify-center items-center text-center cursor-pointer hover:shadow-rose-500/30 transition-all" onClick={() => setIsModalOpen(true)}>
                    <div className="w-12 h-12 bg-white/20 rounded-full flex items-center justify-center mb-2 shadow-sm"><Plus size={24}/></div>
                    <p className="text-[10px] font-black uppercase tracking-widest">New Expense Entry</p>
                </div>
            </div>

            {/* Controls */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div className="flex items-center gap-2 overflow-x-auto max-w-full pb-2 sm:pb-0 no-scrollbar">
                    {['All', ...EXPENSE_CATEGORIES].slice(0, 5).map(cat => (
                        <button
                            key={cat}
                            onClick={() => setCategoryFilter(cat)}
                            className={`px-4 py-2 rounded-xl text-[10px] font-black uppercase tracking-widest transition-all whitespace-nowrap ${categoryFilter === cat ? 'bg-slate-900 text-white shadow-md' : 'bg-white border border-slate-200 text-slate-500 hover:bg-slate-50'}`}
                        >
                            {cat}
                        </button>
                    ))}
                    {(EXPENSE_CATEGORIES.length > 4) && (
                        <div className="relative group">
                            <button className="px-4 py-2 bg-white border border-slate-200 rounded-xl text-[10px] font-black uppercase tracking-widest text-slate-500 hover:bg-slate-50 flex items-center gap-1">
                                <Filter size={12}/> More
                            </button>
                        </div>
                    )}
                </div>

                <div className="relative w-full sm:w-80">
                    <Search className="absolute left-4 top-1/2 -translate-y-1/2 text-slate-400" size={16}/>
                    <input
                        type="text"
                        placeholder="Search description, ref..."
                        className="w-full pl-11 pr-4 py-3 bg-white border border-slate-200 rounded-2xl text-[11px] font-black uppercase tracking-[0.1em] outline-none focus:border-rose-500 shadow-sm transition-all"
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                    />
                </div>
            </div>

            {/* List */}
            <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                <table className="w-full text-left">
                    <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                    <tr>
                        <th className="px-8 py-5">Date & Reference</th>
                        <th className="px-8 py-5">Description</th>
                        <th className="px-8 py-5 text-center">Category</th>
                        <th className="px-8 py-5 text-right">Debit Amount</th>
                    </tr>
                    </thead>
                    <tbody className="divide-y divide-slate-100">
                    {filteredExpenses.map((e: any) => {
                        const accountName = accounts?.find((a:any) => a.id === e.accountId)?.name || 'Unknown Account';
                        return (
                            <tr key={e.id} className="hover:bg-slate-50/50 transition-colors group">
                                <td className="px-8 py-5">
                                    <p className="font-bold text-slate-900 text-xs">{e.date}</p>
                                    <p className="text-[9px] text-slate-400 font-mono mt-1 font-bold uppercase">{e.refNumber || `VCH-${e.id}`}</p>
                                </td>
                                <td className="px-8 py-5">
                                    <p className="font-black text-slate-700 text-xs uppercase">{e.description}</p>
                                    <p className="text-[9px] text-slate-400 font-bold uppercase mt-1 flex items-center gap-1"><Landmark size={10}/> Paid via {accountName}</p>
                                </td>
                                <td className="px-8 py-5 text-center">
                                        <span className={`px-3 py-1 rounded-lg text-[9px] font-black uppercase tracking-wider border ${
                                            ['Rent', 'Salaries'].includes(e.category) ? 'bg-purple-50 text-purple-600 border-purple-100' :
                                                ['Utilities', 'Taxes'].includes(e.category) ? 'bg-amber-50 text-amber-600 border-amber-100' :
                                                    'bg-slate-100 text-slate-500 border-slate-200'
                                        }`}>
                                            {e.category}
                                        </span>
                                </td>
                                <td className="px-8 py-5 text-right">
                                    <p className="font-black font-mono text-sm text-rose-600 flex items-center justify-end gap-1">
                                        <ArrowDownRight size={14}/> {formatCurrency(e.amount)}
                                    </p>
                                </td>
                            </tr>
                        );
                    })}
                    {filteredExpenses.length === 0 && (
                        <tr><td colSpan={4} className="p-12 text-center text-slate-400 text-xs font-bold uppercase italic tracking-widest">No matching expense records found</td></tr>
                    )}
                    </tbody>
                </table>
            </div>

            {/* Create Modal */}
            {isModalOpen && (
                <div className="fixed inset-0 bg-slate-950/80 z-[100] flex items-center justify-center p-4 backdrop-blur-md overflow-y-auto leading-none">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg shadow-2xl border border-white/20 my-auto animate-fade-in relative overflow-hidden">
                        <div className="bg-rose-600 p-8 flex justify-between items-center text-white">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-white/20 rounded-2xl flex items-center justify-center text-white shadow-inner"><Wallet size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Expense Voucher</h3>
                                    <p className="text-[10px] font-bold opacity-80 uppercase tracking-widest mt-1">Record Operational Outflow</p>
                                </div>
                            </div>
                            <button onClick={() => setIsModalOpen(false)} className="p-2 hover:bg-white/20 rounded-full transition-colors"><X size={20} /></button>
                        </div>

                        <div className="p-10 space-y-6">
                            <div className="space-y-4">
                                <IntegratedInput
                                    label="Description / Narration"
                                    value={formData.description}
                                    onChange={(e: any) => setFormData({...formData, description: e.target.value})}
                                    icon={StickyNote}
                                    placeholder="e.g. Shop Electricity Bill"
                                />

                                <div className="grid grid-cols-2 gap-4">
                                    <IntegratedSelect
                                        label="Expense Category"
                                        value={formData.category}
                                        onChange={(e: any) => setFormData({...formData, category: e.target.value})}
                                        options={EXPENSE_CATEGORIES}
                                        icon={Tag}
                                    />
                                    <IntegratedInput
                                        label="Reference #"
                                        value={formData.refNumber}
                                        onChange={(e: any) => setFormData({...formData, refNumber: e.target.value})}
                                        icon={Tag}
                                        placeholder="Receipt No."
                                    />
                                </div>

                                <div className="p-5 bg-slate-50 rounded-2xl border border-slate-100 space-y-4">
                                    <IntegratedInput
                                        label="Amount (PKR)"
                                        value={formData.amount}
                                        onChange={(e: any) => setFormData({...formData, amount: e.target.value})}
                                        icon={DollarSign}
                                    />
                                    <div className="grid grid-cols-2 gap-4">
                                        <IntegratedInput
                                            label="Voucher Date"
                                            type="date"
                                            value={formData.date}
                                            onChange={(e: any) => setFormData({...formData, date: e.target.value})}
                                            icon={Calendar}
                                        />
                                        <IntegratedSelect
                                            label="Source Account"
                                            value={formData.accountId}
                                            onChange={(e: any) => setFormData({...formData, accountId: e.target.value})}
                                            options={accounts}
                                            displayKey="name"
                                            icon={Landmark}
                                        />
                                    </div>
                                </div>
                                <IntegratedInput
                                    label="Additional Notes"
                                    value={formData.notes}
                                    onChange={(e: any) => setFormData({...formData, notes: e.target.value})}
                                    icon={Info}
                                    placeholder="Optional details..."
                                />
                            </div>

                            <button onClick={handleSave} className="w-full bg-slate-900 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-[0.2em] hover:bg-rose-600 transition-all shadow-xl flex items-center justify-center gap-3">
                                <BadgeCheck size={18} /> Authorize Payment
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
