
import React, { useState, useMemo } from 'react';
import {
    StickyNote, DollarSign, Activity, Calendar, Landmark,
    TrendingUp, TrendingDown, Wallet, Plus, ArrowUpRight, ArrowDownRight,
    History, Hash
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export function CapitalPage({ capital, accounts, notify }: any) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [formData, setFormData] = useState({ description: '', amount: '', type: 'Investment', date: getCurrentDate(), accountId: 'ACC-001', refNumber: '' });

    const stats = useMemo(() => {
        const invested = (capital || []).filter((c: any) => c.type === 'Investment').reduce((acc: number, c: any) => acc + (Number(c.amount) || 0), 0);
        const withdrawn = (capital || []).filter((c: any) => c.type === 'Withdrawal').reduce((acc: number, c: any) => acc + (Number(c.amount) || 0), 0);
        return { invested, withdrawn, net: invested - withdrawn };
    }, [capital]);

    const handleSave = () => {
        if (!formData.description || !formData.amount) return;
        router.post('/capital', formData, {
            onSuccess: () => {
                setIsModalOpen(false);
                setFormData({ description: '', amount: '', type: 'Investment', date: getCurrentDate(), accountId: 'ACC-001', refNumber: '' });
                notify('Capital Ledger Updated');
            }
        });
    };

    return (
        <div className="animate-fade-in space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm flex items-center gap-4"><div className="w-12 h-12 rounded-xl bg-emerald-50 text-emerald-600 flex items-center justify-center"><TrendingUp size={24} /></div><div><p className="text-[10px] font-black uppercase text-slate-400 tracking-widest mb-1">Total Investments</p><p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(stats.invested)}</p></div></div>
                <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm flex items-center gap-4"><div className="w-12 h-12 rounded-xl bg-rose-50 text-rose-600 flex items-center justify-center"><TrendingDown size={24} /></div><div><p className="text-[10px] font-black uppercase text-slate-400 tracking-widest mb-1">Total Withdrawals</p><p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(stats.withdrawn)}</p></div></div>
                <div className="bg-slate-900 p-6 rounded-2xl shadow-xl flex items-center gap-4 text-white"><div className="w-12 h-12 rounded-xl bg-white/10 flex items-center justify-center text-cyan-400"><Wallet size={24} /></div><div><p className="text-[10px] font-bold uppercase text-slate-400 tracking-widest mb-1">Net Capital Pool</p><p className="text-xl font-black font-mono">{formatCurrency(stats.net)}</p></div></div>
            </div>

            <div className="flex justify-between items-center px-4 lg:px-0">
                <div className="flex items-center gap-2"><History size={18} className="text-slate-400" /><h3 className="text-lg font-bold font-heading text-slate-900 uppercase italic">Transaction Ledger</h3></div>
                <button onClick={() => setIsModalOpen(true)} className="flex items-center gap-2 px-6 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-bold uppercase tracking-widest shadow-lg hover:bg-black transition-all"><Plus size={14} /> Record Entry</button>
            </div>

            <div className="bg-white rounded-3xl border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                <table className="w-full text-left"><thead className="bg-slate-50 border-b border-slate-100 uppercase italic"><tr className="text-[10px] font-bold text-slate-500"><th className="px-6 py-5">Origin / Description</th><th className="px-6 py-5">Ledger Type</th><th className="px-6 py-5">Date</th><th className="px-6 py-5 text-right">Amount</th></tr></thead>
                    <tbody className="divide-y divide-slate-100">
                    {capital.map((c: any) => (
                        <tr key={c.id} className="hover:bg-slate-50/50 transition-colors">
                            <td className="px-6 py-5"><p className="font-bold text-slate-900 text-sm uppercase">{c.description}</p><p className="text-[10px] text-slate-400 font-bold uppercase mt-1">{c.refNumber ? `Ref: ${c.refNumber}` : ''}</p></td>
                            <td className="px-6 py-5"><span className={`inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-[9px] font-black uppercase border tracking-widest ${c.type === 'Investment' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' : 'bg-rose-50 text-rose-600 border-rose-100'}`}>{c.type === 'Investment' ? <ArrowUpRight size={10} /> : <ArrowDownRight size={10} />}{c.type}</span></td>
                            <td className="px-6 py-5"><p className="text-xs font-mono font-bold text-slate-500">{c.date}</p></td>
                            <td className={`px-6 py-5 text-right font-mono font-black text-sm ${c.type === 'Investment' ? 'text-emerald-600' : 'text-rose-600'}`}>{c.type === 'Investment' ? '+' : '-'}{formatCurrency(c.amount)}</td>
                        </tr>
                    ))}
                    </tbody>
                </table>
            </div>

            {isModalOpen && (
                <div className="fixed inset-0 bg-slate-900/60 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-md p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in">
                        <div className="flex items-center gap-4 border-b border-slate-100 pb-6"><div className="w-12 h-12 bg-slate-100 rounded-2xl flex items-center justify-center text-slate-900"><DollarSign size={24} /></div><div><h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Record Capital</h3></div></div>
                        <div className="space-y-5">
                            <IntegratedSelect label="Entry Direction" value={formData.type} onChange={(e: any) => setFormData({...formData, type: e.target.value})} options={['Investment', 'Withdrawal']} icon={Activity} />
                            <IntegratedInput label="Narration" value={formData.description} onChange={(e: any) => setFormData({...formData, description: e.target.value})} icon={StickyNote} />
                            <div className="grid grid-cols-2 gap-4"><IntegratedInput label="Amount (PKR)" value={formData.amount} onChange={(e: any) => setFormData({...formData, amount: e.target.value})} icon={DollarSign} /><IntegratedInput label="Date" type="date" value={formData.date} onChange={(e: any) => setFormData({...formData, date: e.target.value})} icon={Calendar} /></div>
                            <div className="grid grid-cols-2 gap-4"><IntegratedSelect label="Target Account" value={formData.accountId} onChange={(e: any) => setFormData({...formData, accountId: e.target.value})} options={accounts} displayKey="name" icon={Landmark} /><IntegratedInput label="Ref #" value={formData.refNumber} onChange={(e: any) => setFormData({...formData, refNumber: e.target.value})} icon={Hash} /></div>
                        </div>
                        <button onClick={handleSave} className="w-full bg-slate-900 text-white py-4 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-black transition-all">Commit to Ledger</button>
                    </div>
                </div>
            )}
        </div>
    );
}
