
import React, { useState, useMemo } from 'react';
import {
    Search, ArrowUpRight, ArrowDownLeft, Package, CheckCircle2,
    MoveRight, Truck, Hash, StickyNote, Building2, History,
    ArrowRightLeft, PackageCheck, AlertCircle, X, Calendar, Filter,
    Bike, Info
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export function BikeTransferPage({ transferInventory, transferHistory, notify }: any) {
    const [activeTab, setActiveTab] = useState<'inventory' | 'history'>('inventory');
    const [isReceiveModalOpen, setIsReceiveModalOpen] = useState(false);
    const [isIssueModalOpen, setIsIssueModalOpen] = useState(false);
    const [selectedIds, setSelectedIds] = useState<string[]>([]);
    const [searchTerm, setSearchTerm] = useState('');

    // Forms
    const [receiveForm, setReceiveForm] = useState({
        date: getCurrentDate(),
        source: '',
        refNumber: '',
        notes: '',
        bikeModel: '',
        engine: '',
        chassis: ''
    });

    const [issueForm, setIssueForm] = useState({
        date: getCurrentDate(),
        dealer: '',
        refNumber: '',
        notes: ''
    });

    // --- 1. DATA NORMALIZATION LAYER (The Fix) ---
    // This bridges the gap between DB (snake_case) and UI (camelCase)
    const normalizedInventory = useMemo(() => {
        return (transferInventory || []).map((i: any) => ({
            ...i,
            id: i.id,
            // Map DB columns to UI props safely
            engine: i.engine || i.engine_number || '---',
            chassis: i.chassis || i.chassis_number || '---',
            date: i.date || i.entry_date || (i.created_at ? i.created_at.split('T')[0] : getCurrentDate()),
            source: i.source || 'Transfer Pool',
            model: i.model || i.details || (i.notes ? i.notes.split('|')[0] : 'Unknown Model')
        }));
    }, [transferInventory]);

    const normalizedHistory = useMemo(() => {
        return (transferHistory || []).map((h: any) => ({
            ...h,
            id: h.id,
            type: h.type || 'UNKNOWN',
            party: h.party || 'Unknown Party',
            // Map DB columns to UI props safely
            refNumber: h.refNumber || h.ref_number || '---',
            date: h.date || (h.created_at ? h.created_at.split('T')[0] : getCurrentDate()),
            engine: h.engine || h.engine_number || '---',
            details: h.details || 'Unit Transfer'
        }));
    }, [transferHistory]);

    // --- 2. Stats Calculation (Using Normalized Data) ---
    const stats = useMemo(() => {
        const poolSize = normalizedInventory.length;
        const today = getCurrentDate();
        const inboundToday = normalizedHistory.filter((h:any) => h.type === 'IN' && h.date === today).length;
        const outboundToday = normalizedHistory.filter((h:any) => h.type === 'OUT' && h.date === today).length;
        return { poolSize, inboundToday, outboundToday };
    }, [normalizedInventory, normalizedHistory]);

    // --- 3. Filtering (Using Normalized Data + Safe Strings) ---
    const filteredInventory = useMemo(() => {
        if (!searchTerm) return normalizedInventory;
        const lower = searchTerm.toLowerCase();
        return normalizedInventory.filter((i: any) =>
            String(i.engine).toLowerCase().includes(lower) ||
            String(i.chassis).toLowerCase().includes(lower) ||
            String(i.model).toLowerCase().includes(lower) ||
            String(i.source).toLowerCase().includes(lower)
        );
    }, [normalizedInventory, searchTerm]);

    const filteredHistory = useMemo(() => {
        if (!searchTerm) return normalizedHistory;
        const lower = searchTerm.toLowerCase();
        return normalizedHistory.filter((h: any) =>
            String(h.refNumber).toLowerCase().includes(lower) ||
            String(h.party).toLowerCase().includes(lower) ||
            String(h.details).toLowerCase().includes(lower) ||
            String(h.engine).toLowerCase().includes(lower)
        );
    }, [normalizedHistory, searchTerm]);

    // --- Handlers ---
    const handleReceiveSubmit = () => {
        if(!receiveForm.engine || !receiveForm.source) return;

        const payload = {
            ...receiveForm,
            items: [{
                engine: receiveForm.engine,
                chassis: receiveForm.chassis
            }]
        };

        router.post('/transfers/receive', payload, {
            onSuccess: () => {
                setIsReceiveModalOpen(false);
                notify('Transit Stock Received');
                setReceiveForm({ date: getCurrentDate(), source: '', refNumber: '', notes: '', bikeModel: '', engine: '', chassis: '' });
            }
        });
    };

    const handleIssueSubmit = () => {
        if(!issueForm.dealer || selectedIds.length === 0) return;
        router.post('/transfers/issue', { ...issueForm, ids: selectedIds }, {
            onSuccess: () => {
                setIsIssueModalOpen(false);
                setSelectedIds([]);
                notify(`${selectedIds.length} Units Dispatched`);
                setIssueForm({ date: getCurrentDate(), dealer: '', refNumber: '', notes: '' });
            }
        });
    };

    const toggleSelection = (id: string) => setSelectedIds(prev => prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]);
    const clearSelection = () => setSelectedIds([]);

    return (
        <div className="animate-fade-in space-y-8 pb-10">
            {/* KPI Dashboard */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl flex items-center gap-5 relative overflow-hidden group">
                    <div className="absolute right-0 top-0 p-4 opacity-10 group-hover:scale-110 transition-transform"><Package size={80}/></div>
                    <div className="w-14 h-14 bg-blue-600 rounded-2xl flex items-center justify-center text-white shadow-lg"><ArrowRightLeft size={28}/></div>
                    <div>
                        <p className="text-[10px] font-black text-blue-300 uppercase tracking-widest mb-1">Transit Holding</p>
                        <h3 className="text-3xl font-black font-mono tracking-tight">{stats.poolSize} <span className="text-sm font-bold text-slate-400">Units</span></h3>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-emerald-50 text-emerald-600 rounded-xl flex items-center justify-center shadow-inner"><ArrowDownLeft size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Today's Arrival</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{stats.inboundToday}</p>
                        <p className="text-[8px] font-bold text-emerald-500 uppercase">External Logistics</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-orange-50 text-orange-600 rounded-xl flex items-center justify-center shadow-inner"><ArrowUpRight size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Today's Dispatch</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{stats.outboundToday}</p>
                        <p className="text-[8px] font-bold text-orange-500 uppercase">Forwarded Units</p>
                    </div>
                </div>
            </div>

            {/* Controls */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div className="flex gap-2 p-1.5 bg-white border border-slate-200 rounded-xl shadow-sm">
                    <button onClick={() => setActiveTab('inventory')} className={`flex items-center gap-2 px-6 py-2.5 rounded-lg text-[10px] font-black uppercase tracking-widest transition-all ${activeTab === 'inventory' ? 'bg-slate-900 text-white shadow-md' : 'text-slate-400 hover:bg-slate-50'}`}>
                        <Package size={14}/> Transit Pool
                    </button>
                    <button onClick={() => setActiveTab('history')} className={`flex items-center gap-2 px-6 py-2.5 rounded-lg text-[10px] font-black uppercase tracking-widest transition-all ${activeTab === 'history' ? 'bg-slate-900 text-white shadow-md' : 'text-slate-400 hover:bg-slate-50'}`}>
                        <History size={14}/> Logistics Log
                    </button>
                </div>

                <div className="flex gap-3 w-full sm:w-auto">
                    <div className="relative flex-1 sm:w-64">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={14} />
                        <input type="text" placeholder="Search Transit Units..." className="w-full pl-9 pr-4 py-2.5 bg-white border border-slate-200 rounded-xl text-xs font-bold uppercase tracking-widest outline-none focus:border-blue-500 transition-all shadow-sm" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                    </div>
                    {activeTab === 'inventory' && (
                        <>
                            <button onClick={() => setIsReceiveModalOpen(true)} className="px-4 py-2.5 bg-emerald-600 text-white rounded-xl text-[10px] font-black uppercase tracking-widest shadow-lg hover:bg-emerald-700 transition-all flex items-center gap-2">
                                <ArrowDownLeft size={14}/> Log Arrival
                            </button>
                            {selectedIds.length > 0 && (
                                <button onClick={() => setIsIssueModalOpen(true)} className="px-4 py-2.5 bg-orange-600 text-white rounded-xl text-[10px] font-black uppercase tracking-widest shadow-lg hover:bg-orange-700 transition-all flex items-center gap-2 animate-fade-in">
                                    <ArrowUpRight size={14}/> Dispatch ({selectedIds.length})
                                </button>
                            )}
                        </>
                    )}
                </div>
            </div>

            {/* Inventory Table */}
            {activeTab === 'inventory' && (
                <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                    {selectedIds.length > 0 && (
                        <div className="bg-blue-50 px-6 py-3 flex justify-between items-center border-b border-blue-100">
                            <span className="text-[10px] font-black text-blue-600 uppercase tracking-widest flex items-center gap-2">
                                <CheckCircle2 size={14}/> {selectedIds.length} Units Selected for Dispatch
                            </span>
                            <button onClick={clearSelection} className="text-[9px] font-bold text-blue-400 hover:text-blue-700 uppercase">Clear Selection</button>
                        </div>
                    )}
                    <table className="w-full text-left">
                        <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                        <tr>
                            <th className="px-6 py-5">Selection</th>
                            <th className="px-6 py-5">Unit Identification</th>
                            <th className="px-6 py-5">Description & Source</th>
                            <th className="px-6 py-5 text-right">Transit Status</th>
                        </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100">
                        {filteredInventory.map((i: any) => (
                            <tr
                                key={i.id}
                                onClick={() => toggleSelection(i.id)}
                                className={`cursor-pointer transition-all ${selectedIds.includes(i.id) ? 'bg-blue-50/50' : 'hover:bg-slate-50'}`}
                            >
                                <td className="px-6 py-5">
                                    <div className={`w-5 h-5 rounded-lg border-2 flex items-center justify-center transition-all ${selectedIds.includes(i.id) ? 'bg-blue-600 border-blue-600 text-white' : 'border-slate-300 bg-white'}`}>
                                        {selectedIds.includes(i.id) && <CheckCircle2 size={12}/>}
                                    </div>
                                </td>
                                <td className="px-6 py-5">
                                    <div className="flex items-center gap-3">
                                        <div className="p-2 bg-slate-100 rounded-lg text-slate-400"><Hash size={14}/></div>
                                        <div>
                                            <p className="text-xs font-black uppercase text-slate-900">{i.engine}</p>
                                            <p className="text-[9px] font-bold uppercase text-slate-400 font-mono mt-0.5">CHS: {i.chassis}</p>
                                        </div>
                                    </div>
                                </td>
                                <td className="px-6 py-5">
                                    <div className="flex flex-col">
                                        <span className="text-[10px] font-black uppercase text-slate-900">{i.model}</span>
                                        <div className="flex items-center gap-2 mt-1">
                                            <Building2 size={10} className="text-slate-400"/>
                                            <span className="text-[9px] font-bold uppercase text-slate-500">From: {i.source}</span>
                                        </div>
                                    </div>
                                </td>
                                <td className="px-6 py-5 text-right">
                                    <div className="flex flex-col items-end">
                                        <span className="px-3 py-1 rounded-full bg-blue-50 border border-blue-100 text-[9px] font-bold text-blue-600 uppercase tracking-widest">In Holding</span>
                                        <span className="text-[8px] font-bold text-slate-400 uppercase mt-1">{i.date}</span>
                                    </div>
                                </td>
                            </tr>
                        ))}
                        {filteredInventory.length === 0 && (
                            <tr>
                                <td colSpan={4} className="p-12 text-center">
                                    <div className="flex flex-col items-center gap-3 text-slate-300">
                                        <Package size={32} />
                                        <p className="text-xs font-bold uppercase tracking-widest">Transit Pool Empty</p>
                                    </div>
                                </td>
                            </tr>
                        )}
                        </tbody>
                    </table>
                </div>
            )}

            {/* History Table */}
            {activeTab === 'history' && (
                <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                    <table className="w-full text-left">
                        <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                        <tr>
                            <th className="px-6 py-5">Transaction Type</th>
                            <th className="px-6 py-5">Logistics Partner</th>
                            <th className="px-6 py-5">Unit Detail</th>
                            <th className="px-6 py-5 text-right">Date & Ref</th>
                        </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100">
                        {filteredHistory.map((h: any, idx: number) => (
                            <tr key={idx} className="hover:bg-slate-50 transition-colors">
                                <td className="px-6 py-5">
                                        <span className={`inline-flex items-center gap-2 px-3 py-1.5 rounded-lg text-[9px] font-black uppercase tracking-widest border ${
                                            h.type === 'IN' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' : 'bg-orange-50 text-orange-600 border-orange-100'
                                        }`}>
                                            {h.type === 'IN' ? <ArrowDownLeft size={12}/> : <ArrowUpRight size={12}/>}
                                            {h.type === 'IN' ? 'Received' : 'Dispatched'}
                                        </span>
                                </td>
                                <td className="px-6 py-5">
                                    <p className="text-xs font-bold uppercase text-slate-900">{h.party}</p>
                                    <p className="text-[9px] font-bold text-slate-400 uppercase mt-0.5">Dealer / Vendor</p>
                                </td>
                                <td className="px-6 py-5">
                                    <p className="text-[10px] font-bold uppercase text-slate-600 truncate max-w-[200px]">{h.details}</p>
                                    <p className="text-[8px] font-mono uppercase text-slate-400 mt-0.5">ENG: {h.engine}</p>
                                </td>
                                <td className="px-6 py-5 text-right">
                                    <p className="text-[10px] font-bold text-slate-900">{h.date}</p>
                                    <p className="text-[9px] font-mono text-slate-400 font-bold uppercase mt-0.5">{h.refNumber}</p>
                                </td>
                            </tr>
                        ))}
                        {filteredHistory.length === 0 && (
                            <tr>
                                <td colSpan={4} className="p-12 text-center text-slate-400 text-xs font-bold uppercase italic">No logistics logs found</td>
                            </tr>
                        )}
                        </tbody>
                    </table>
                </div>
            )}

            {/* RECEIVE MODAL */}
            {isReceiveModalOpen && (
                <div className="fixed inset-0 bg-slate-900/80 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex justify-between items-center border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-emerald-600 rounded-2xl flex items-center justify-center text-white"><ArrowDownLeft size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Inbound Logistics</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1">Receive Unit (Non-Purchase)</p>
                                </div>
                            </div>
                            <button onClick={() => setIsReceiveModalOpen(false)} className="p-2 hover:bg-slate-50 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="space-y-5">
                            {/* Visual Warning for Non-Stock */}
                            <div className="flex items-center gap-3 p-4 bg-emerald-50 border border-emerald-100 rounded-2xl">
                                <Info size={16} className="text-emerald-600 shrink-0"/>
                                <p className="text-[9px] font-bold text-emerald-700 uppercase leading-relaxed">
                                    Units entered here are tracked as <strong>Third-Party Transit</strong>.
                                    They will <span className="underline">not</span> appear in Sales Inventory or Financial Reports.
                                </p>
                            </div>

                            <IntegratedInput label="Source / Sender Name" value={receiveForm.source} onChange={(e: any) => setReceiveForm({...receiveForm, source: e.target.value})} icon={Building2} placeholder="e.g. Lahore Center" />
                            <IntegratedInput label="Vehicle Model / Description" value={receiveForm.bikeModel} onChange={(e: any) => setReceiveForm({...receiveForm, bikeModel: e.target.value})} icon={Bike} placeholder="e.g. Yamaha YBR 125 (Black)" />

                            <div className="p-5 bg-slate-50 rounded-2xl border border-slate-100 space-y-4">
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest border-b border-slate-200 pb-2 mb-2">Unit Identifiers</p>
                                <div className="grid grid-cols-1 gap-4">
                                    <IntegratedInput label="Engine Number" value={receiveForm.engine} onChange={(e: any) => setReceiveForm({...receiveForm, engine: e.target.value})} icon={Hash} />
                                    <IntegratedInput label="Chassis Number" value={receiveForm.chassis} onChange={(e: any) => setReceiveForm({...receiveForm, chassis: e.target.value})} icon={Hash} />
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Gate Pass Ref" value={receiveForm.refNumber} onChange={(e: any) => setReceiveForm({...receiveForm, refNumber: e.target.value})} icon={StickyNote} />
                                <IntegratedInput label="Date" type="date" value={receiveForm.date} onChange={(e: any) => setReceiveForm({...receiveForm, date: e.target.value})} icon={Calendar} />
                            </div>
                        </div>
                        <button onClick={handleReceiveSubmit} className="w-full bg-emerald-600 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-emerald-700 transition-all flex items-center justify-center gap-2">
                            <CheckCircle2 size={16}/> Confirm Receipt
                        </button>
                    </div>
                </div>
            )}

            {/* ISSUE MODAL */}
            {isIssueModalOpen && (
                <div className="fixed inset-0 bg-slate-900/80 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-xl p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex justify-between items-center border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-orange-600 rounded-2xl flex items-center justify-center text-white"><ArrowUpRight size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Outbound Dispatch</h3>
                                    <p className="text-[10px] font-bold text-orange-500 uppercase mt-1">{selectedIds.length} Units Selected for Issue</p>
                                </div>
                            </div>
                            <button onClick={() => setIsIssueModalOpen(false)} className="p-2 hover:bg-slate-50 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="space-y-5">
                            {/* Selected Units Preview */}
                            <div className="bg-slate-50 p-4 rounded-2xl border border-slate-100 max-h-32 overflow-y-auto custom-scrollbar">
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-2 sticky top-0 bg-slate-50">Manifest Preview</p>
                                <div className="space-y-1">
                                    {selectedIds.map(id => {
                                        const item = normalizedInventory.find((i:any) => i.id === id);
                                        return (
                                            <div key={id} className="flex justify-between items-center text-[10px] border-b border-slate-200 pb-1 last:border-0">
                                                <span className="font-bold text-slate-700">{item?.engine}</span>
                                                <span className="font-mono text-slate-400 truncate w-32 text-right">{item?.model || item?.chassis}</span>
                                            </div>
                                        );
                                    })}
                                </div>
                            </div>

                            <IntegratedInput label="Destination Dealer / Receiver" value={issueForm.dealer} onChange={(e: any) => setIssueForm({...issueForm, dealer: e.target.value})} icon={Truck} placeholder="Authorized Dealer Name" />

                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Gate Pass / Ref" value={issueForm.refNumber} onChange={(e: any) => setIssueForm({...issueForm, refNumber: e.target.value})} icon={Hash} />
                                <IntegratedInput label="Date" type="date" value={issueForm.date} onChange={(e: any) => setIssueForm({...issueForm, date: e.target.value})} icon={Calendar} />
                            </div>
                            <IntegratedInput label="Logistics Notes" value={issueForm.notes} onChange={(e: any) => setIssueForm({...issueForm, notes: e.target.value})} icon={StickyNote} placeholder="Driver name, Vehicle No, etc." />
                        </div>
                        <button onClick={handleIssueSubmit} className="w-full bg-slate-900 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-black transition-all flex items-center justify-center gap-2">
                            <PackageCheck size={16}/> Authorize Dispatch
                        </button>
                    </div>
                </div>
            )}
        </div>
    );
}
