import React, { useState, useEffect } from 'react';
import { Head, Link, useForm } from '@inertiajs/react';
import {
    Eye, EyeOff, Power, Loader2, Zap, Lock,
    Phone, Cpu, Settings, Activity, Thermometer, ShieldCheck
} from 'lucide-react';

// --- STYLES & ANIMATIONS ---
const customStyles = `
@import url('https://fonts.googleapis.com/css2?family=Exo+2:ital,wght@0,400;0,600;0,700;0,800;1,700&family=Rajdhani:wght@500;600;700&display=swap');
.bg-carbon {
    background-color: #020617;
    background-image: linear-gradient(45deg, #0f172a 25%, transparent 25%, transparent 75%, #0f172a 75%, #0f172a),
                      linear-gradient(45deg, #0f172a 25%, transparent 25%, transparent 75%, #0f172a 75%, #0f172a);
    background-size: 4px 4px;
    background-position: 0 0, 2px 2px;
}
.font-sport { font-family: 'Exo 2', sans-serif; }
.font-hud { font-family: 'Rajdhani', sans-serif; }
.clip-aero { clip-path: polygon(15px 0, 100% 0, 100% calc(100% - 15px), calc(100% - 15px) 100%, 0 100%, 0 15px); }
.clip-button { clip-path: polygon(10% 0, 100% 0, 100% 70%, 90% 100%, 0 100%, 0 30%); }
@keyframes speed-pass { 0% { transform: translateX(-100%); } 100% { transform: translateX(100%); } }
.animate-speed-pass { animation: speed-pass linear infinite; }
`;

// --- MINI COMPONENTS ---
const RPMGauge = ({ value }: { value: number }) => {
    const percentage = Math.min(value / 30, 1);
    const dashArray = `${percentage * 126} 126`;
    return (
        <div className="relative w-16 h-16 flex items-center justify-center">
            <svg viewBox="0 0 50 50" className="w-full h-full transform -rotate-90">
                <circle cx="25" cy="25" r="20" fill="none" stroke="#1e293b" strokeWidth="4" />
                <circle cx="25" cy="25" r="20" fill="none" stroke="#3b82f6" strokeWidth="4"
                        strokeDasharray={dashArray} strokeLinecap="round" className="transition-all duration-500" />
            </svg>
            <div className="absolute inset-0 flex flex-col items-center justify-center">
                <span className="text-[10px] font-bold font-hud text-blue-400">{Math.round(percentage * 100)}%</span>
            </div>
        </div>
    );
};

export default function Login({ status }: { status?: string }) {
    const [showPassword, setShowPassword] = useState(false);
    const [mousePos, setMousePos] = useState({ x: 0, y: 0 });
    const [fieldFocus, setFieldFocus] = useState<string | null>(null);

    // 1. Inertia Form Integration
    const { data, setData, post, processing, errors, reset } = useForm({
        email: '',
        password: '',
        remember: false,
    });

    const handleMouseMove = (e: React.MouseEvent) => {
        setMousePos({
            x: (e.clientX / window.innerWidth - 0.5) * 15,
            y: (e.clientY / window.innerHeight - 0.5) * 15,
        });
    };

    const submit = (e: React.FormEvent) => {
        e.preventDefault();
        post(route('login'), {
            onFinish: () => reset('password'),
        });
    };

    return (
        <div className="min-h-screen flex items-center justify-center p-6 relative overflow-hidden bg-[#020617] text-white font-sport" onMouseMove={handleMouseMove}>
            <style>{customStyles}</style>
            <Head title="System Ignition" />

            {/* Background FX */}
            <div className="absolute inset-0 bg-carbon opacity-40 z-0" />
            <div className="absolute top-0 left-1/2 -translate-x-1/2 w-[80vw] h-[50vh] bg-blue-900/20 rounded-full blur-[120px] pointer-events-none" />

            {/* Left Column: Branding */}
            <div className="w-full max-w-[950px] relative z-10 grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                <div className="hidden lg:block" style={{ transform: `translate(${mousePos.x * -0.5}px, ${mousePos.y * -0.5}px)` }}>
                    <h1 className="text-7xl font-black italic tracking-tighter leading-none">
                        MOTO<br/><span className="text-transparent bg-clip-text bg-gradient-to-r from-blue-400 to-cyan-300">SHOW</span>
                    </h1>
                    <div className="mt-6 flex gap-4">
                        <div className="px-4 py-2 border-l-2 border-blue-500 bg-slate-900/50 backdrop-blur-sm">
                            <div className="text-[10px] text-slate-400 font-hud uppercase">System Status</div>
                            <div className="text-blue-400 font-bold">OPERATIONAL</div>
                        </div>
                    </div>
                </div>

                {/* Right Column: Form */}
                <div className="w-full max-w-[420px] mx-auto relative group" style={{ transform: `translate(${mousePos.x * 0.5}px, ${mousePos.y * 0.5}px)` }}>
                    <div className="relative bg-[#0f172a] border-y border-[#1e293b] shadow-2xl overflow-hidden clip-aero ring-1 ring-white/5">

                        {/* Success Overlay when Processing */}
                        {processing && (
                            <div className="absolute inset-0 z-50 bg-[#0f172a]/90 backdrop-blur-md flex flex-col items-center justify-center">
                                <Loader2 className="animate-spin text-blue-500 mb-2" size={40} />
                                <span className="font-hud text-blue-400 tracking-widest text-xs">IGNITING...</span>
                            </div>
                        )}

                        <div className="p-8">
                            <div className="flex gap-4 text-[10px] font-hud text-slate-500 mb-6 border-b border-white/5 pb-2">
                                <span>OIL_PSI: <span className="text-white">42.5</span></span>
                                <span className="ml-auto text-blue-400">SYS_READY</span>
                            </div>

                            <form onSubmit={submit} className="space-y-6">
                                {/* Email Field */}
                                <div className="space-y-1">
                                    <label className="text-[10px] font-bold uppercase tracking-widest text-slate-400">Access Key (Email)</label>
                                    <div className="relative bg-[#020617] skew-x-[-10deg] border border-slate-700 focus-within:border-blue-500 transition-all">
                                        <input
                                            type="email"
                                            value={data.email}
                                            onChange={e => setData('email', e.target.value)}
                                            className="w-full bg-transparent border-none text-white px-4 py-3 outline-none font-hud text-lg skew-x-[10deg] focus:ring-0"
                                            placeholder="ENTER EMAIL"
                                        />
                                    </div>
                                    {errors.email && <div className="text-red-400 text-[10px] font-bold mt-1 uppercase italic">{errors.email}</div>}
                                </div>

                                {/* Password Field */}
                                <div className="space-y-1">
                                    <div className="flex justify-between">
                                        <label className="text-[10px] font-bold uppercase tracking-widest text-slate-400">Security Code</label>
                                        <button type="button" onClick={() => setShowPassword(!showPassword)} className="text-[10px] text-blue-400 font-bold uppercase">Toggle</button>
                                    </div>
                                    <div className="relative bg-[#020617] skew-x-[-10deg] border border-slate-700 focus-within:border-blue-500 transition-all">
                                        <input
                                            type={showPassword ? "text" : "password"}
                                            value={data.password}
                                            onChange={e => setData('password', e.target.value)}
                                            className="w-full bg-transparent border-none text-white px-4 py-3 outline-none font-hud text-lg skew-x-[10deg] focus:ring-0"
                                            placeholder="••••••••"
                                        />
                                    </div>
                                    {errors.password && <div className="text-red-400 text-[10px] font-bold mt-1 uppercase italic">{errors.password}</div>}
                                </div>

                                {/* Button Section */}
                                <div className="flex items-center gap-4">
                                    <button type="submit" disabled={processing} className="flex-1 relative group overflow-hidden py-5 px-6 clip-button">
                                        <div className="absolute inset-0 bg-gradient-to-r from-blue-700 to-blue-500 group-hover:from-blue-600 transition-colors" />
                                        <div className="relative flex items-center justify-between z-10">
                                            <span className="text-xl font-black italic text-white tracking-wide">IGNITION</span>
                                            <Power size={20} className={processing ? 'animate-pulse' : ''} />
                                        </div>
                                    </button>
                                    <div className="hidden sm:block"><RPMGauge value={data.email.length + data.password.length} /></div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            {/* Footer Bar */}
            <div className="absolute bottom-0 w-full bg-[#020617]/90 backdrop-blur-md border-t border-white/10 p-6 flex justify-between items-center z-20">
                <div className="flex items-center gap-4">
                    <div className="w-10 h-10 bg-slate-800 clip-button flex items-center justify-center border border-white/10"><Zap size={18} className="text-blue-400 fill-current" /></div>
                    <div>
                        <div className="text-[10px] text-slate-500 font-hud uppercase">Developer Terminal</div>
                        <div className="text-sm font-black text-white italic">MR. ABBAS <span className="text-blue-400 font-normal ml-2">v4.2.0</span></div>
                    </div>
                </div>
                <div className="text-right">
                    <div className="text-[10px] text-slate-500 font-hud uppercase tracking-widest">Support Uplink</div>
                    <div className="text-sm font-black text-white font-mono tracking-wide">0319 433 433 5</div>
                </div>
            </div>
        </div>
    );
}
