<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{

    public function down(): void
    {
        Schema::dropIfExists('claims');
        Schema::dropIfExists('workshops');
    }


    public function up(): void
    {
        // 1. Workshops Table
        Schema::create('workshops', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('contact_person')->nullable();
            $table->string('phone')->nullable();
            $table->text('address')->nullable();
            $table->string('status')->default('Active');

            // Financials
            $table->decimal('balance', 15, 2)->default(0); // Track total payable to workshop

            $table->timestamps();
        });

        // 2. Claims Table
        Schema::create('claims', function (Blueprint $table) {
            $table->id();
            $table->string('type')->default('Sold Unit'); // 'Sold Unit', 'External Service'
            $table->string('sale_id')->nullable(); // UUID from sales table (if Sold Unit)
            $table->unsignedBigInteger('workshop_id')->nullable(); // Assigned Workshop
            $table->unsignedBigInteger('supplier_id')->nullable(); // Supplier responsible for warranty

            // Snapshot details
            $table->string('customer_name')->nullable();
            $table->string('engine_number')->nullable();
            $table->text('description')->nullable();

            // Financials
            // workshop_cost = Payable (Liability)
            $table->decimal('workshop_cost', 15, 2)->default(0);
            // claimed_amount = Receivable (Asset)
            $table->decimal('claimed_amount', 15, 2)->default(0);

            // Lifecycle Status
            $table->string('status')->default('Pending'); // Pending, Completed
            $table->string('workshop_status')->default('Pending'); // Pending, Paid
            $table->string('supplier_status')->default('Pending'); // Pending, Invoiced, Recovered

            $table->date('claim_date');
            $table->timestamps();

            $table->foreign('workshop_id')->references('id')->on('workshops')->nullOnDelete();
            $table->foreign('supplier_id')->references('id')->on('suppliers')->nullOnDelete();
        });

        // 3. Ensure vendor_payments can handle workshops (Optional: add workshop_id if not polymorphic)
        // For this implementation, we will assume vendor_payments has a generic structure or we add workshop_id column
        // If vendor_payments is strictly for suppliers, we can add a nullable workshop_id to it.
        Schema::table('vendor_payments', function (Blueprint $table) {
            $table->unsignedBigInteger('workshop_id')->nullable()->after('supplier_id');
            $table->unsignedBigInteger('supplier_id')->nullable()->change(); // Make supplier nullable if paying workshop
        });
    }


};
