<?php
namespace App\Http\Controllers; // Ensure Namespace
use Illuminate\Http\Request;
use Illuminate\Support\Str; // Import Str
use Illuminate\Support\Facades\DB;
use App\Models\{Sale, StockUnit, Payment, Account, Booking};

class TerminalController extends Controller {
public function storeSale(Request $request) {
$request->validate([
'customerId' => 'required',
'engineNumber' => 'required|exists:stock_units,engine_number',
'totalPrice' => 'required|numeric'
]);

DB::transaction(function() use ($request) {
// 1. Lock the unit to prevent double selling
$unit = StockUnit::where('engine_number', $request->engineNumber)->lockForUpdate()->first();

// FIX: Explicitly check if unit exists before accessing status to prevent 500 Error
if (!$unit) {
throw \Illuminate\Validation\ValidationException::withMessages([
'engineNumber' => ['Stock unit not found or currently unavailable.']
]);
}

if ($unit->status !== 'In Stock') {
throw new \Exception('Unit already sold');
}

$saleId = (string) Str::uuid(); // Explicit UUID generation

$sale = Sale::create([
'id' => $saleId,
'customer_id' => $request->customerId,
'stock_unit_id' => $unit->id,
'sale_date' => $request->date,
'cost_price' => $unit->purchase_price,
'total_amount' => $request->totalPrice,
// FIX: Default null/empty values to 0 to prevent SQL Integrity Violation (Column cannot be null)
'down_payment' => $request->downPayment ?? 0,
'payment_type' => $request->paymentType,
'installments_count' => $request->installmentsCount ?? 0,
'monthly_installment' => $request->monthlyAmount ?? 0,
'notes' => $request->notes,
'ref_number' => $request->refNumber
]);

$unit->update(['status' => 'Sold']);

// Handle Down Payment
if (($request->downPayment ?? 0) > 0) {
Payment::create([
'sale_id' => $saleId,
'account_id' => $request->accountId,
'amount' => $request->downPayment,
'payment_date' => $request->date,
'type' => 'DownPayment'
]);
Account::find($request->accountId)->increment('balance', $request->downPayment);
}
});

return redirect()->back()->with('message', 'Sale Invoiced');
}

public function storePayment(Request $request) {
DB::transaction(function() use ($request) {
// 1. Record Payment
Payment::create([
'sale_id' => $request->saleId,
'account_id' => $request->accountId,
'amount' => $request->amount,
'payment_date' => $request->date,
'type' => 'Installment',
'note' => $request->note
]);

// 2. Update Account
Account::find($request->accountId)->increment('balance', $request->amount);

// 3. Check for Settlement (Enhancement)
$sale = Sale::lockForUpdate()->find($request->saleId);
$totalPaid = Payment::where('sale_id', $request->saleId)->sum('amount');

if ($totalPaid >= $sale->total_amount) {
// Mark sale as fully settled/closed if needed
// Assuming 'status' column exists on sales table.
// Currently it might be 'Active'. We can change to 'Settled'.
// If the frontend relies on paymentType !== 'Full Payment' to determine lease,
// we don't strictly need to change status, but it's good practice.
$sale->update(['status' => 'Settled']);
}
});
return redirect()->back()->with('message', 'Payment Received');
}

public function storeBooking(Request $request) {
DB::transaction(function() use ($request) {
$booking = Booking::updateOrCreate(['id' => $request->id], [
'customer_id' => $request->customerId,
'product_id' => $request->productId,
'total' => $request->totalPrice,
'advance' => $request->advanceAmount,
'status' => $request->status,
'booking_date' => $request->date,
'promised_date' => $request->promisedDate,
'notes' => $request->notes,
'ref_number' => $request->refNumber,
'account_id' => $request->accountId // Store account reference if column exists, or rely on Payment record
]);

// AUTOMATED FINANCIAL RECORDING
// Only creating payment for NEW bookings with advance > 0 to avoid duplicates on edits.
if ($booking->wasRecentlyCreated && $request->advanceAmount > 0 && $request->accountId) {

// 1. Create Payment Record (Linked to Booking)
// Note: Ensure Payment model 'sale_id' is nullable or you have a 'booking_id' column
// OR store it with sale_id = null and put Booking Ref in notes.
Payment::create([
'sale_id' => null,
'account_id' => $request->accountId,
'amount' => $request->advanceAmount,
'payment_date' => $request->date,
'type' => 'Booking Advance',
'note' => "Advance for Booking ID: " . $booking->id
]);

// 2. Update Account Balance
Account::find($request->accountId)->increment('balance', $request->advanceAmount);
}
});

return redirect()->back()->with('message', 'Booking Saved & Ledger Updated');
}

public function fulfillBooking(Request $request) {
$request->validate([
'bookingId' => 'required|exists:bookings,id',
'engineNumber' => 'required|exists:stock_units,engine_number',
'settlementAmount' => 'nullable|numeric|min:0',
'accountId' => 'required_if:settlementAmount,>0|exists:accounts,id'
]);

DB::transaction(function() use ($request) {
// 1. Get Booking
$booking = Booking::lockForUpdate()->find($request->bookingId);
if($booking->status === 'Fulfilled') {
throw new \Exception('Booking already fulfilled');
}

// 2. Lock Stock
$unit = StockUnit::where('engine_number', $request->engineNumber)->lockForUpdate()->first();
if (!$unit || $unit->status !== 'In Stock') {
throw \Illuminate\Validation\ValidationException::withMessages([
'engineNumber' => ['Selected unit is not available.']
]);
}

// 3. Create Sale
$saleId = (string) Str::uuid();
$totalDown = $request->totalDownPayment ?? ($booking->advance + ($request->settlementAmount ?? 0));

$sale = Sale::create([
'id' => $saleId,
'customer_id' => $booking->customer_id,
'stock_unit_id' => $unit->id,
'sale_date' => $request->date, // Delivery Date
'cost_price' => $unit->purchase_price,
'total_amount' => $request->totalPrice ?? $booking->total,
'down_payment' => $totalDown,
'payment_type' => $request->paymentType,
'installments_count' => $request->installmentsCount ?? 0,
'monthly_installment' => $request->monthlyAmount ?? 0,
'notes' => "Fulfilled Booking " . $booking->ref_number . ". " . $request->notes,
'ref_number' => $request->refNumber,
'status' => 'Active'
]);

// 4. Update States
$unit->update(['status' => 'Sold']);
$booking->update(['status' => 'Fulfilled']);

// 5. Record Financials (ONLY New Cash)
// The advance is already in the system (ledgered when booking was made).
// We only record the *Settlement Amount* collected now.
$settlement = $request->settlementAmount ?? 0;
if ($settlement > 0) {
Payment::create([
'sale_id' => $saleId,
'account_id' => $request->accountId,
'amount' => $settlement,
'payment_date' => $request->date,
'type' => 'DownPayment', // Or 'Settlement'
'note' => 'Booking Settlement'
]);
Account::find($request->accountId)->increment('balance', $settlement);
}
});

return redirect()->back()->with('message', 'Booking Fulfilled & Inventory Updated');
}
}
