<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{FeeBatch, Account, Payment};

class FeeController extends Controller
{
    /**
     * Dispatch a new batch of service coupons
     */
    public function store(Request $request)
    {
        $request->validate([
            'supplierId' => 'required|exists:suppliers,id',
            'qty' => 'required|numeric|min:1',
            'rate' => 'required|numeric|min:0',
            'date' => 'required|date'
        ]);

        $expected = $request->qty * $request->rate;

        FeeBatch::create([
            'supplier_id' => $request->supplierId,
            'dispatch_date' => $request->date,
            'qty' => $request->qty,
            'rate' => $request->rate,
            'courier_ref' => $request->courierRef,
            'expected_amount' => $expected,
            'status' => 'Pending',
            'notes' => $request->notes
        ]);

        return back()->with('message', 'Service Batch Dispatched');
    }

    /**
     * Receive payment for a batch (Supports Partial Payments)
     */
    public function settle(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id',
            'date' => 'required|date'
        ]);

        DB::transaction(function () use ($request, $id) {
            $batch = FeeBatch::lockForUpdate()->findOrFail($id);

            // 1. Record Income (Cash Inflow treated as Profit)
            // We pass 'sale_id' => null because this is income from a supplier, not a customer sale.
            Payment::create([
                'sale_id' => null,
                'account_id' => $request->accountId,
                'amount' => $request->amount,
                'payment_date' => $request->date,
                'type' => 'Fee Recovery',
                'note' => "Fee Reimbursement Batch " . substr($batch->id, 0, 8) . ($request->notes ? " | " . $request->notes : "")
            ]);

            Account::find($request->accountId)->increment('balance', $request->amount);

            // 2. Update Batch Status
            $totalReceived = $batch->received_amount + $request->amount;

            // Use round to avoid floating point precision errors
            $remaining = round($batch->expected_amount - $totalReceived, 2);

            $status = $remaining <= 0 ? 'Cleared' : 'Partial';

            $batch->update([
                'received_amount' => $totalReceived,
                'status' => $status
            ]);
        });

        return back()->with('message', 'Payment Recorded & Ledger Updated');
    }
}
