<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\DB;
use App\Models\{
    Product, Customer, Supplier, StockUnit, Sale, Expense,
    Booking, Account, CapitalEntry, FeeBatch, Registration,
    Claim, Workshop, TransferLog, DealerExchange, BusinessAdvance,
    ProductAttribute
};

class DashboardController extends Controller
{
    public function index()
    {
        // 1. Fetch Explicit Attributes
        $attributes = ProductAttribute::all();

        // 2. Fetch Legacy Attributes (distinct values currently used in products)
        $legacyMakes = Product::distinct()->pluck('make')->filter();
        $legacyYears = Product::distinct()->pluck('year')->filter();
        $legacyColors = Product::distinct()->pluck('color')->filter();

        // 3. Merge, Unique, and Sort
        $makes = collect($attributes->where('type', 'make')->pluck('value'))
            ->merge($legacyMakes)->unique()->sort()->values();

        $years = collect($attributes->where('type', 'year')->pluck('value'))
            ->merge($legacyYears)->unique()->sort()->reverse()->values();

        $colors = collect($attributes->where('type', 'color')->pluck('value'))
            ->merge($legacyColors)->unique()->sort()->values();

        return Inertia::render('App', [
            'products' => Product::all(),
            'customers' => Customer::all(),
            'suppliers' => Supplier::all(),
            'accounts' => Account::all(),
            'physicalStock' => StockUnit::orderBy('entry_date', 'desc')->get(),
            'purchaseOrders' => \App\Models\PurchaseOrder::with('supplier')->orderBy('date', 'desc')->get(),
            'salesHistory' => Sale::with(['customer', 'payments'])->orderBy('sale_date', 'desc')->get(),
            'bookings' => Booking::orderBy('booking_date', 'desc')->get(),
            'expenses' => Expense::orderBy('expense_date', 'desc')->get(),
            'capital' => CapitalEntry::orderBy('entry_date', 'desc')->get(),
            'registrations' => Registration::orderBy('registration_date', 'desc')->get(),
            'claims' => Claim::orderBy('claim_date', 'desc')->get(),
            'feeBatches' => FeeBatch::orderBy('dispatch_date', 'desc')->get(),
            'workshops' => Workshop::all(),
            'transferInventory' => StockUnit::where('status', 'Transfer Pool')->get()->map(function($u) { $u->source = 'Transfer Pool'; return $u; }),
            'transferHistory' => TransferLog::orderBy('date', 'desc')->limit(100)->get(),
            'dealerExchanges' => DealerExchange::orderBy('created_at', 'desc')->get(),
            'vendorPayments' => DB::table('vendor_payments')->orderBy('payment_date', 'desc')->get(),
            'advances' => BusinessAdvance::orderBy('created_at', 'desc')->get(), // Critical for B2B Loans module

            // Meta
            'makes' => $makes,
            'years' => $years,
            'colors' => $colors,
            'settings' => config('showroom.settings', [
                'showroomName' => 'MOTOAPP',
                'address' => 'Main Boulevard',
                'hiddenModules' => []
            ]),
        ]);
    }
}
