<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inertia\Inertia;
use App\Models\{BusinessAdvance, Account, Payment};

class BusinessAdvanceController extends Controller
{
    public function index()
    {
        return Inertia::render('BusinessAdvancesPage', [
            'advances' => BusinessAdvance::orderBy('created_at', 'desc')->get(),
            'accounts' => Account::all()
        ]);
    }

    /**
     * Issue Money (Lend)
     */
    public function store(Request $request)
    {
        $request->validate([
            'partner_name' => 'required|string',
            'amount' => 'required|numeric|min:1',
            'account_id' => 'required|exists:accounts,id',
            'issue_date' => 'required|date'
        ]);

        DB::transaction(function () use ($request) {
            // 1. Create the Advance Record
            $advance = BusinessAdvance::create([
                'partner_name' => $request->partner_name,
                'contact_info' => $request->contact,
                'principal_amount' => $request->amount,
                'agreed_return_amount' => $request->expected_return, // Optional
                'issue_date' => $request->issue_date,
                'due_date' => $request->due_date,
                'notes' => $request->notes,
                'ref_number' => $request->ref_number,
                'status' => 'Active'
            ]);

            // 2. Deduct Money from Account (This is an Asset Transfer, not an Expense)
            // We verify balance first
            $account = Account::lockForUpdate()->find($request->account_id);
            if ($account->balance < $request->amount) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'amount' => ['Insufficient funds in selected account.']
                ]);
            }

            $account->decrement('balance', $request->amount);

            // 3. Log the Outflow in Vendor Payments (or a generic transaction ledger)
            // Using DB table insert for speed, referencing the Advance ID
            DB::table('vendor_payments')->insert([
                'account_id' => $request->account_id,
                'amount' => $request->amount,
                'payment_date' => $request->issue_date,
                'notes' => "Business Advance to {$request->partner_name}",
                'ref_number' => "ADV-" . substr($advance->id, 0, 8),
                'created_at' => now(),
                'updated_at' => now()
            ]);
        });

        return back()->with('message', 'Business Advance Issued Successfully');
    }

    /**
     * Receive Money Back (Return + Profit)
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1', // Total amount being returned right now
            'account_id' => 'required|exists:accounts,id',
            'date' => 'required|date'
        ]);

        DB::transaction(function () use ($request, $id) {
            $advance = BusinessAdvance::lockForUpdate()->findOrFail($id);

            // 1. Calculate Financials
            // We treat the return as filling the Principal bucket first.
            // Any amount exceeding the remaining principal is considered Profit.

            $remainingPrincipal = $advance->principal_amount - $advance->returned_amount;
            $currentReturn = $request->amount;

            $principalCovered = 0;
            $profitGenerated = 0;

            if ($currentReturn <= $remainingPrincipal) {
                // Not even covering full principal yet
                $principalCovered = $currentReturn;
            } else {
                // Covered remaining principal, rest is profit
                $principalCovered = $remainingPrincipal; // (can be 0 if already covered)
                $profitGenerated = $currentReturn - $principalCovered;
            }

            // 2. Record Income/Inflow
            Payment::create([
                'sale_id' => null,
                'account_id' => $request->account_id,
                'amount' => $request->amount,
                'payment_date' => $request->date,
                'type' => $profitGenerated > 0 ? 'Investment Return & Profit' : 'Principal Recovery',
                'note' => "Return from {$advance->partner_name}. Principal: {$principalCovered}, Profit: {$profitGenerated}"
            ]);

            // 3. Update Account
            Account::find($request->account_id)->increment('balance', $request->amount);

            // 4. Update Advance Record
            $newReturned = $advance->returned_amount + $principalCovered; // Only track principal repayment here
            // Note: We track profit separately or just via returned_total logic depending on preference.
            // Here, let's track total cash back vs principal.

            // Re-evaluating logic: simpler to track Total Returned and define status.
            $totalReturnedSoFar = $advance->returned_amount + $request->amount; // This mixes principal and profit

            // Let's stick to the schema: returned_amount tracks RAW CASH BACK.
            // Profit is calculated as (returned_amount - principal_amount) if positive.

            $newStatus = $advance->status;
            if ($request->is_final_settlement) {
                $newStatus = 'Settled';
                // Calculate final realized profit
                $finalProfit = max(0, $totalReturnedSoFar - $advance->principal_amount);
                $advance->profit_realized = $finalProfit;
            } elseif ($totalReturnedSoFar >= $advance->principal_amount) {
                $newStatus = 'Profitable'; // We have our money back, everything else is gravy
            }

            $advance->update([
                'returned_amount' => $totalReturnedSoFar,
                'status' => $newStatus,
                'profit_realized' => max(0, $totalReturnedSoFar - $advance->principal_amount)
            ]);
        });

        return back()->with('message', 'Return Received Successfully');
    }
}
